/*
 * Copyright 2012 Stefan Haun, Andreas Nürnberger
 * 
 *      Data and Knowledge Engineering Group, 
 * 		Faculty of Computer Science,
 *		Otto-von-Guericke University,
 *		Magdeburg, Germany
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.ovgu.dke.mocca.util;

import java.net.URI;
import java.util.HashMap;
import java.util.Map;

import net.jcip.annotations.ThreadSafe;

import de.ovgu.dke.mocca.api.context.Context;
import de.ovgu.dke.mocca.api.context.ContextConnectionListener;

/**
 * Directory of context associated to a key object, i.e. a GLUE packet thread.
 * 
 * @author Stefan Haun (stefan.haun@ovgu.de)
 * 
 */
@ThreadSafe
public enum ContextDirectory implements ContextConnectionListener {
	INSTANCE;

	@Deprecated
	public static synchronized ContextDirectory getInstance() {
		return INSTANCE;
	}

	private final Map<Object, Context> contexts;

	private ContextDirectory() {
		this.contexts = new HashMap<Object, Context>();
	}

	/**
	 * Store a context by its ID.
	 * 
	 * @param id
	 *            The reference ID.
	 * @param ctx
	 *            The context to be stored.
	 * @throws NullPointerException
	 *             if the id argument is null.
	 */
	public void store(Object id, Context ctx) {
		synchronized (this.contexts) {
			this.contexts.put(id, ctx);
			ctx.putAttribute(Context.ATTR_ID, id.toString());
		}
	}

	/**
	 * Retrieve a context by its ID.
	 * 
	 * @param id
	 *            The reference ID.
	 * @return The context, if found, otherwise null.
	 * @throws NullPointerException
	 *             if the id argument is null.
	 */
	public Context retrieve(Object id) {
		synchronized (this.contexts) {
			return this.contexts.get(id);
		}
	}

	/**
	 * Remove the context stored for id.
	 * 
	 * @param id
	 *            The reference ID.
	 * @throws NullPointerException
	 *             if the id argument is null.
	 */
	public void remove(Object id) {
		synchronized (this.contexts) {
			this.contexts.remove(id);
		}
	}

	@Override
	public void contextConnected(Context ctx, URI peer) {
		// TODO Auto-generated method stub

	}

	@Override
	public void contextDisconnected(Context ctx, URI peer) {
		// TODO Auto-generated method stub

	}

	@Override
	public void contextDialogOpen(Context ctx, String id) {
		this.store(id, ctx);
	}

	@Override
	public void contextDialogClose(Context ctx, String id) {
		this.remove(id);
	}
}
