/*
 * Copyright 2012 Stefan Haun, Andreas Nürnberger
 * 
 *      Data and Knowledge Engineering Group, 
 * 		Faculty of Computer Science,
 *		Otto-von-Guericke University,
 *		Magdeburg, Germany
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.ovgu.dke.mocca.util;

import java.net.URI;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import net.jcip.annotations.ThreadSafe;
import de.ovgu.dke.mocca.api.command.CommandHandler;
import de.ovgu.dke.mocca.api.command.CommandHandlerRegistry;

/**
 * Support implementation of a command handler registry.
 * 
 * @author Stefan Haun (stefan.haun@ovgu.de)
 */
@ThreadSafe
public class CommandHandlerSupport implements CommandHandlerRegistry {
	private final Map<URI, CommandHandler> handlers;

	public CommandHandlerSupport() {
		this.handlers = new HashMap<URI, CommandHandler>();
	}

	@Override
	public void registerCommandHandler(CommandHandler hnd) {
		if (hnd == null)
			throw new NullPointerException(
					"The handler argument must not be null!");

		synchronized (handlers) {
			final Collection<URI> commands = hnd.getAvailableCommands();
			if (commands != null)
				for (URI command : commands)
					this.handlers.put(command, hnd);
		}
	}

	@Override
	public void removeCommandHandler(URI command) {
		if (command == null)
			throw new NullPointerException(
					"The command argument must not be null!");

		synchronized (handlers) {
			this.handlers.remove(command);
		}
	}

	@Override
	public CommandHandler getCommandHandler(URI command) {
		if (command == null)
			throw new NullPointerException(
					"The command argument must not be null!");

		synchronized (handlers) {
			return this.handlers.get(command);
		}
	}

}
