/*
 * Copyright 2012 Stefan Haun, Andreas Nürnberger
 * 
 *      Data and Knowledge Engineering Group, 
 * 		Faculty of Computer Science,
 *		Otto-von-Guericke University,
 *		Magdeburg, Germany
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.ovgu.dke.mocca.glue;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Properties;

import de.ovgu.dke.glue.api.serialization.SerializationException;
import de.ovgu.dke.glue.api.serialization.SerializationProvider;
import de.ovgu.dke.glue.util.serialization.TextSerializationHelpers;
import de.ovgu.dke.mocca.api.command.Command;
import de.ovgu.dke.mocca.api.command.CommandException;
import de.ovgu.dke.mocca.api.command.CommandFactory;
import de.ovgu.dke.mocca.impl.DefaultCommandFactory;
import de.ovgu.dke.mocca.util.MoccaHelper;

/**
 * <p>
 * Serializer for a Command, which serializes to Text returned as
 * <code>String</code>.
 * </p>
 * 
 * @author Stefan Haun (stefan.haun@ovgu.de)
 * 
 */
public class TextCommandSerializer extends CommandSerializer {
	@Override
	public Object serialize(Object o) throws SerializationException {
		// check object type
		if (!(o instanceof Command))
			throw new ClassCastException(
					"Serializer expected Command implementation, got "
							+ (o == null ? "null" : o.getClass()
									.getCanonicalName()) + " instead!");

		// convert to command
		final Command cmd = (Command) o;

		// here goes the result
		final StringBuilder text = new StringBuilder();

		// first line: command
		if (cmd.getCommand() == null)
			throw new SerializationException(
					"Command did not contain a command identifier!");
		text.append(cmd.getCommand().toASCIIString());

		final Properties props = MoccaHelper.cmdParams2Props(cmd);
		final String s_props = TextSerializationHelpers.encodeProperties(props);
		if (!s_props.isEmpty()) {
			// newline
			text.append("\n");

			// parameters
			text.append(s_props);
		}

		// return a String
		return text.toString();
	}

	@Override
	public Object deserialize(Object o) throws SerializationException {
		final String text = (String) o;

		// first line is the command
		int _cmdidx = text.indexOf('\n');
		String _command = _cmdidx < 0 ? text : text.substring(0, _cmdidx);
		// create the URI
		final URI command;
		try {
			command = new URI(_command);
		} catch (URISyntaxException e) {
			throw new SerializationException("Identifier URI is invalid: "
					+ e.getMessage(), e);
		}

		// the rest is parameters, if available
		final Properties params;
		if (_cmdidx > 0) {
			final String _params = text.substring(_cmdidx + 1);
			params = TextSerializationHelpers.decodeProperties(_params);
		} else
			params = new Properties();

		// create and return the command
		final CommandFactory fac = new DefaultCommandFactory();
		try {
			return fac.createCommand(command, params);
		} catch (CommandException e) {
			throw new SerializationException(
					"Error creating command instance: " + e.getMessage(), e);
		}
	}

	@Override
	public String getFormat() {
		return SerializationProvider.STRING;
	}
}
