/*
 * Copyright 2012 Stefan Haun, Andreas Nürnberger
 * 
 *      Data and Knowledge Engineering Group, 
 * 		Faculty of Computer Science,
 *		Otto-von-Guericke University,
 *		Magdeburg, Germany
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.ovgu.dke.mocca.glue;

import java.net.URI;
import java.util.Properties;

import de.ovgu.dke.glue.api.serialization.SerializationProvider;
import de.ovgu.dke.glue.api.transport.PacketHandlerFactory;
import de.ovgu.dke.glue.api.transport.SchemaRecord;
import de.ovgu.dke.glue.api.transport.SchemaRegistry;
import de.ovgu.dke.glue.api.transport.TransportException;
import de.ovgu.dke.glue.api.transport.TransportRegistry;
import de.ovgu.dke.glue.util.serialization.SingleSerializerProvider;
import de.ovgu.dke.mocca.api.MoccaException;
import de.ovgu.dke.mocca.api.MoccaRuntime;
import de.ovgu.dke.mocca.api.command.Command;
import de.ovgu.dke.mocca.api.command.CommandFactory;
import de.ovgu.dke.mocca.api.command.CommandHandlerRegistry;
import de.ovgu.dke.mocca.api.context.Context;
import de.ovgu.dke.mocca.impl.DefaultCommandFactory;
import de.ovgu.dke.mocca.util.CommandHandlerSupport;
import de.ovgu.dke.mocca.util.ContextDirectory;

/**
 * GLUE implementation of a MOCCA runtime.
 * 
 * @author Stefan Haun (stefan.haun@ovgu.de)
 * 
 */
public class GlueMoccaRuntimeImpl implements MoccaRuntime {
	private final CommandHandlerRegistry globalHandlers;
	private final CommandFactory commandFactory;

	public GlueMoccaRuntimeImpl() {
		this.globalHandlers = new CommandHandlerSupport();
		this.commandFactory = new DefaultCommandFactory();
	}

	@Override
	public void init(final Properties env) throws MoccaException {
		final SerializationProvider serializers = SingleSerializerProvider
				.of(new TextCommandSerializer());

		try {
			final PacketHandlerFactory packetHandlers = CommandPacketHandlerFactory
					.forRuntime(this);

			// initialize the transports
			// TODO move this to an exterinal initialization.
			TransportRegistry.getInstance().loadTransportFactory(
					"de.ovgu.dke.glue.xmpp.transport.XMPPTransportFactory",
					env, TransportRegistry.AS_DEFAULT,
					TransportRegistry.DEFAULT_KEY);

			// register MOCCA as middleware
			final SchemaRecord schema = SchemaRecord.valueOf(
					CommandSerializer.SCHEMA, packetHandlers, serializers);
			SchemaRegistry.getInstance().registerSchemaRecord(schema);

		} catch (TransportException e) {
			throw new MoccaException("Error initializing the GLUE transport: "
					+ e.getMessage(), e);
		} catch (ClassNotFoundException e) {
			throw new MoccaException(
					"Could not load the XMPP transport class!", e);
		}

	}

	@Override
	public void dispose() throws MoccaException {
		// dispose the transport factory
		TransportRegistry.getInstance().disposeAll();
	}

	@Override
	public Context createContext() throws MoccaException {
		return GlueContextImpl.create(
				TransportRegistry.getDefaultTransportFactory(),
				ContextDirectory.INSTANCE);
	}

	@Override
	public Command createCommand(final URI command, final Properties parameters)
			throws MoccaException {
		return this.commandFactory.createCommand(command, parameters);
	}

	@Override
	public CommandHandlerRegistry getCommandHandlerRegistry() {
		return globalHandlers;
	}

}
