/*
 * Copyright 2012 Stefan Haun, Andreas Nürnberger
 * 
 *      Data and Knowledge Engineering Group, 
 * 		Faculty of Computer Science,
 *		Otto-von-Guericke University,
 *		Magdeburg, Germany
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.ovgu.dke.mocca.glue;

import java.net.URI;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import net.jcip.annotations.ThreadSafe;
import de.ovgu.dke.glue.api.transport.Connection;
import de.ovgu.dke.glue.api.transport.Packet;
import de.ovgu.dke.glue.api.transport.PacketThread;
import de.ovgu.dke.glue.api.transport.Transport;
import de.ovgu.dke.glue.api.transport.TransportException;
import de.ovgu.dke.glue.api.transport.TransportFactory;
import de.ovgu.dke.glue.api.transport.TransportRegistry;
import de.ovgu.dke.mocca.api.MoccaException;
import de.ovgu.dke.mocca.api.MoccaRuntime;
import de.ovgu.dke.mocca.api.command.Command;
import de.ovgu.dke.mocca.api.command.CommandHandlerRegistry;
import de.ovgu.dke.mocca.api.context.Context;
import de.ovgu.dke.mocca.api.context.ContextConnectionListener;
import de.ovgu.dke.mocca.api.context.State;
import de.ovgu.dke.mocca.util.CommandHandlerSupport;
import de.ovgu.dke.mocca.util.ContextConnectionSupport;
import de.ovgu.dke.mocca.util.MoccaHelper;
import de.ovgu.dke.mocca.util.UpdatableState;

/**
 * Implementation for the Context interface using GLUE.
 * 
 * @author Stefan Haun (stefan.haun@ovgu.de)
 */
@ThreadSafe
public class GlueContextImpl implements Context {
	private final TransportFactory transports;

	private URI peer = null;
	private Connection connection = null;
	private PacketThread thread = null;

	private final Map<String, Object> attributes = new ConcurrentHashMap<String, Object>();
	private final CommandHandlerSupport handlers = new CommandHandlerSupport();

	private final ContextConnectionSupport connectionListeners = new ContextConnectionSupport();

	/**
	 * Create a GLUE context for a specific transport factory.
	 * 
	 * @param transports
	 *            The transport factory.
	 * @param listener
	 *            A context connection listener, null to ignore the listener.
	 *            Adding the listener here allows to catch events fired during
	 *            creation.
	 * @return An new GLUE context instance.
	 * @throws NullPointerException
	 *             if the transport argument is null.
	 */
	public static GlueContextImpl create(final TransportFactory transports,
			final ContextConnectionListener listener) {
		final GlueContextImpl ctx = new GlueContextImpl(transports);

		// register listener, if available
		if (listener != null)
			ctx.addContextConnectionListener(listener);

		return ctx;
	}

	/**
	 * Create a GLUE context from an incoming packet thread.
	 * 
	 * @param pt
	 *            The packet thread to bind this context to.
	 * @param listener
	 *            A context connection listener, null to ignore the listener.
	 *            Adding the listener here allows to catch events fired during
	 *            creation.
	 * @return An new GLUE context instance.
	 * @throws NullPointerException
	 *             if the transport argument is null.
	 */
	public static GlueContextImpl createFromPacketThread(final PacketThread pt,
			final ContextConnectionListener listener) {
		if (pt == null)
			throw new NullPointerException(
					"Packet Thread parameter may not be null!");

		if (pt.getConnection() == null)
			throw new NullPointerException("Packet thread has no connection!");

		// TODO use the right transport factory from pt
		GlueContextImpl ctx = GlueContextImpl.create(
				TransportRegistry.getDefaultTransportFactory(), listener);

		ctx.thread = pt;
		ctx.connection = pt.getConnection();
		ctx.peer = pt.getConnection().getPeer();

		if (listener != null) {
			listener.contextConnected(ctx, ctx.peer);
			// TODO use thread id, when available
			listener.contextDialogOpen(ctx, ((Object) pt).toString());
		}
		return ctx;
	}

	/**
	 * Private constructor, use @code{create} or @code{createFromPacketThread}
	 * instead!
	 */
	private GlueContextImpl(final TransportFactory transports) {
		if (transports == null)
			throw new NullPointerException(
					"Transports parameter may not be null!");

		this.transports = transports;
	}

	private TransportFactory getTransports() {
		return this.transports;
	}

	@Override
	public void connect(URI peer) throws MoccaException {
		synchronized (this.transports) {
			if (peer == null) {
				disconnect();
				return;
			}

			// get a transport
			try {
				final Transport transport = getTransports().createTransport(
						peer);

				// get the connection
				this.connection = transport
						.getConnection(CommandSerializer.SCHEMA);

				if (!connection.checkCapabilities()) {
					throw new MoccaException(
							"Cannot establish MOCCA communication with "
									+ peer.toASCIIString());
				}

				this.peer = peer;

				connectionListeners.contextConnected(this, this.peer);

			} catch (TransportException e) {
				throw new MoccaException("Error creating GLUE transport: "
						+ e.getMessage(), e);
			}
		}
	}

	@Override
	public void disconnect() throws MoccaException {
		synchronized (this.transports) {
			// GLUE disconnect
			if (thread != null) {
				// TODO use thread id, when available
				connectionListeners.contextDialogClose(this, thread.toString());
				thread.dispose();
			}
			this.connection = null;

			connectionListeners.contextDisconnected(this, this.peer);

			this.peer = null;
		}
	}

	@Override
	public URI getPeer() {
		synchronized (this.transports) {
			return this.peer;
		}
	}

	@Override
	public void sendCommand(Command cmd) throws MoccaException {
		synchronized (this.transports) {
			if (connection == null)
				throw new MoccaException("Context is not connected!");

			try {
				// create a packet thread, if not available
				if (thread == null) {
					thread = connection
							.createThread(PacketThread.DEFAULT_HANDLER);
					// TODO use thread id, when available
					connectionListeners.contextDialogOpen(this,
							thread.toString());
				}

				thread.send(cmd, Packet.Priority.DEFAULT);
			} catch (TransportException e) {
				throw new MoccaException("Error sending command via GLUE: "
						+ e.getMessage(), e);
			}
		}
	}

	@Override
	public void putAttribute(String key, Object value) {
		if (value == null)
			this.attributes.remove(key);
		else
			this.attributes.put(key, value);
	}

	@Override
	public Object getAttribute(String key) {
		return this.attributes.get(key);
	}

	@Override
	public Set<String> getAttributes() {
		// TODO unmodifiable?
		return this.attributes.keySet();
	}

	@Override
	public MoccaRuntime getRuntime() {
		return MoccaHelper.getDefaultRuntime();
	}

	@Override
	public CommandHandlerRegistry getCommandHandlerRegistry() {
		return handlers;
	}

	@Override
	public State getState() throws MoccaException {
		synchronized (attributes) {
			Object _state = this.getAttribute(ATTR_STATE);

			if (_state != null && !(_state instanceof State))
				throw new MoccaException("The key " + ATTR_STATE
						+ " did not provide a state object in this context, "
						+ "something else has been put there!");

			if (_state == null) {
				_state = new UpdatableState();
				this.putAttribute(ATTR_STATE, _state);
			}
			return (State) _state;
		}
	}

	@Override
	public void addContextConnectionListener(ContextConnectionListener listener) {
		connectionListeners.addListener(listener);
	}

	@Override
	public void removeContextConnectoinListener(
			ContextConnectionListener listener) {
		connectionListeners.removeListener(listener);
	}

}
