/*
 * Copyright 2012 Stefan Haun, Andreas Nürnberger
 * 
 *      Data and Knowledge Engineering Group, 
 * 		Faculty of Computer Science,
 *		Otto-von-Guericke University,
 *		Magdeburg, Germany
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.ovgu.dke.mocca.glue;

import de.ovgu.dke.glue.api.transport.PacketHandler;
import de.ovgu.dke.glue.api.transport.PacketHandlerFactory;
import de.ovgu.dke.mocca.api.MoccaRuntime;

/**
 * Packet handler factory for the MOCCA Command Packets.
 * 
 * @author Stefan Haun (stefan.haun@ovgu.de)
 */
public class CommandPacketHandlerFactory implements PacketHandlerFactory {
	private final MoccaRuntime runtime;

	/**
	 * Get a command packet handler factory for a specific runtime.
	 * 
	 * @param runtime
	 *            The MOCCA runtime connected to this command packet handler
	 *            factory.
	 * @return A command packet handler factory.
	 * @throws NullPointerException
	 *             if the runtime argument is null.
	 */
	public static PacketHandlerFactory forRuntime(MoccaRuntime runtime) {
		return new CommandPacketHandlerFactory(runtime);
	}

	/**
	 * Instantiate a command packet handler factory connected to a specific
	 * MOCCA runtime.
	 * 
	 * @param runtime
	 *            The MOCCA runtime connected to this command packet handler
	 *            factory.
	 * @throws NullPointerException
	 *             if the runtime argument is null.
	 */
	private CommandPacketHandlerFactory(MoccaRuntime runtime) {
		if (runtime == null)
			throw new NullPointerException(
					"The runtime argument must not be null!");
		this.runtime = runtime;
	}

	/**
	 * Create a packet handler.
	 * 
	 * @return A new packet handler instance.
	 * @throws InstantiationException
	 *             if the packet handler cannot be instantiated.
	 */
	@Override
	public PacketHandler createPacketHandler() throws InstantiationException {
		return new CommandPacketHandler(runtime);
	}
}
