/*
 * Copyright 2012 Stefan Haun, Andreas Nürnberger
 * 
 *      Data and Knowledge Engineering Group, 
 * 		Faculty of Computer Science,
 *		Otto-von-Guericke University,
 *		Magdeburg, Germany
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.ovgu.dke.mocca.api.context;

import java.net.URI;

import de.ovgu.dke.mocca.api.MoccaException;
import de.ovgu.dke.mocca.api.command.Command;

/**
 * The context state.
 * 
 * @author Stefan Haun (stefan.haun@ovgu.de)
 *
 */
public interface State {
	/**
	 * URI prefix for this element
	 */
	public static final String PREFIX = "http://dke.ovgu.de/mocca/core/state";

	/**
	 * The attribute key for storing in MOCCA Context
	 */
	public static final String ATTR = PREFIX;

	/**
	 * The command URI for context state updates.
	 */
	public static final URI CMD_UPDATE = URI.create(PREFIX + "/update");

	/**
	 * Parameter URI for the message text value.
	 */
	public static final String PARA_TEXT = PREFIX + "#text";
	/**
	 * Parameter URI for the life-cycle value.
	 */
	public static final String PARA_LIFECYCLE = PREFIX + "#lifecycle";
	/**
	 * Parameter URI for the minimal progress value.
	 */
	public static final String PARA_MIN = PREFIX + "#min";
	/**
	 * Parameter URI for the maximal progress value.
	 */
	public static final String PARA_MAX = PREFIX + "#max";
	/**
	 * Parameter URI for the current progress value.
	 */
	public static final String PARA_CURRENT = PREFIX + "#current";
	/**
	 * Parameter URI for the message severity value.
	 */
	public static final String PARA_SEVERITY = PREFIX + "#severity";
	/**
	 * Parameter URI for the trace description value.
	 */
	public static final String PARA_TRACE = PREFIX + "#trace";

	/**
	 * Parameter URI denoting that a value should be removed.
	 */
	public static final String NIL = "#nil";

	/**
	 * Context Life-Cycle State
	 */
	public enum Lifecycle {
		IDLE, RUNNING, FAILED, CLOSED;
	}

	/**
	 * State Message Severity
	 */
	public enum Severity {
		DEBUG, INFO, WARN, ERROR, FATAL;
	}

	/**
	 * Get the text value.
	 * 
	 * The text is a short representation of the message's intent, often a
	 * keyword representing a state.
	 * 
	 * @return the value of the text property; null if none has been set.
	 */
	public String getText();

	/**
	 * Set the text value.
	 * 
	 * The text is a short representation of the message's intent, often a
	 * keyword representing a state.
	 * 
	 * @param text
	 *            the new value of the text property, null to delete it.
	 */
	public void setText(final String text);

	/**
	 * Get the life-cycle value.
	 * 
	 * The life-cycle represents the state of the context.
	 * 
	 * @return the value of the life-cycle property, one of {@link Lifecycle};
	 *         null if none has been set.
	 */
	public Lifecycle getLifecycle();

	/**
	 * Set the life-cycle value.
	 * 
	 * The life-cycle represents the state of the context.
	 * 
	 * @param lifecycle
	 *            the new value of the life-cycle property, null to delete it.
	 */
	public void setLifecycle(final Lifecycle lifecycle);

	/**
	 * Get the minimal progress value.
	 * 
	 * @return the value of the minimal-progress property, null if none has been
	 *         set.
	 */
	public Integer getMinProgress();

	/**
	 * Set the minimal progress value.
	 * 
	 * @param progress_min
	 *            the minimal progress value, null to delete it.
	 */
	public void setMinProgress(final Integer minProgress);

	/**
	 * Get the maximal progress value.
	 * 
	 * @return the value of the maximal-progress property, null if none has been
	 *         set.
	 */
	public Integer getMaxProgress();

	/**
	 * Set the maximal progress value.
	 * 
	 * @param progress_max
	 *            The new value of the maximal-progres property, null to delete
	 *            it.
	 */
	public void setMaxProgress(final Integer maxProgress);

	/**
	 * Get the current progress value.
	 * 
	 * @return The value of the current-progress property, null if none has been
	 *         set.
	 */
	public Integer getCurrentProgress();

	/**
	 * Set the current progress value.
	 * 
	 * @param progress_current
	 *            The new value of the current-progress property, null to delete
	 *            it.
	 */
	public void setCurrentProgress(final Integer currentProgress);

	/**
	 * Get the severity of the state.
	 * 
	 * The severity can be used for log filtering and a generic handling of
	 * status updates.
	 * 
	 * @return The value of the severity property, null if none has been set.
	 */
	public Severity getSeverity();

	/**
	 * Set the severity of the state.
	 * 
	 * The severity can be used for log filtering and a generic handling of
	 * status updates.
	 * 
	 * @param severity
	 *            The new value for the severity property, null to delete it.
	 */
	public void setSeverity(final Severity severity);

	/**
	 * Get the trace information of the state.
	 * 
	 * This property can be used to provide a more detailed description of the
	 * state, e.g. a Stack Trace from an {@link Exception}.
	 * 
	 * @return the value of the trace property, null if none has been set.
	 */
	public String getTrace();

	/**
	 * Set the trace information of the state.
	 * 
	 * This property can be used to provide a more detailed description of the
	 * state, e.g. a Stack Trace from an {@link Exception}.
	 * 
	 * @param trace
	 *            The new value of the trace property, null to delete it.
	 */
	public void setTrace(final String trace);

	/**
	 * Create an update-status command containing only the properties that have
	 * been changed since the last status update.
	 * 
	 * @param ctx
	 *            The context in which the command should be created.
	 * @return an update-status command with the changed status properties or
	 *         null, if the status is unchanged.
	 * @throws MoccaException
	 *             if command creation fails.
	 * @throws NullPointerException
	 *             if the context argument is null.
	 */
	public Command createStateUpdate(final Context ctx) throws MoccaException;
}
