/*
 * Copyright 2012 Stefan Haun, Andreas Nürnberger
 * 
 *      Data and Knowledge Engineering Group, 
 * 		Faculty of Computer Science,
 *		Otto-von-Guericke University,
 *		Magdeburg, Germany
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.ovgu.dke.mocca.api.context;

import java.net.URI;
import java.util.Set;

import net.jcip.annotations.ThreadSafe;
import de.ovgu.dke.mocca.api.MoccaException;
import de.ovgu.dke.mocca.api.MoccaRuntime;
import de.ovgu.dke.mocca.api.command.Command;
import de.ovgu.dke.mocca.api.command.CommandHandlerRegistry;

/**
 * Mocca context with attributes and control methods for sending messages and
 * context status.
 * 
 * @author Stefan Haun (stefan.haun@ovgu.de)
 */
@ThreadSafe
public interface Context {
	/**
	 * The context directory stores the context id as String representation here
	 */
	public static final String ATTR_ID = "http://dke.ovgu.de/mocca/core/Context#id";

	/**
	 * The context state is accessible via this key
	 */
	public static final String ATTR_STATE = "http://dke.ovgu.de/mocca/core/Context#state";

	/**
	 * Connect this context to a peer. A context can only be connected to one
	 * peer at a time.
	 * 
	 * @param peer
	 *            The peer's URI
	 * @throws MoccaException
	 *             if the connection fails
	 */
	public void connect(final URI peer) throws MoccaException;

	/**
	 * Disconnect from a peer
	 * 
	 * @throws MoccaException
	 *             if the context cannot be disconnected.
	 */
	public void disconnect() throws MoccaException;

	/**
	 * Get the peer this context is currently connected to.
	 * 
	 * @return Peer URI or <code>null</code> if the context is not connected.
	 */
	public URI getPeer();

	/**
	 * Send a command to the current peer.
	 * 
	 * @param cmd
	 *            The command to be sent.
	 * @throws MoccaException
	 *             if sending fails, i.e. the context is not connected.
	 */
	public void sendCommand(final Command cmd) throws MoccaException;

	/**
	 * Put an attribute into this context, which can be retrieved by subsequent
	 * command handlers.
	 * 
	 * @param key
	 *            The key to the attribute.
	 * @param value
	 *            The attribute value, <code>null</code> to delete the
	 *            attribute.
	 */
	public void putAttribute(final String key, final Object value);

	/**
	 * Retrieve an attribute from the context.
	 * 
	 * @param key
	 *            The key to the attribute.
	 * @return The attribute value or <code>null</code> if no such attribute
	 *         could be found.
	 */
	public Object getAttribute(final String key);

	/**
	 * Get all stored attribute keys.
	 * 
	 * @return Set of keys from the attribute map.
	 */
	public Set<String> getAttributes();

	/**
	 * Get the Mocca Runtime for this context. Currently this is the default
	 * runtime, however the framework may support different Runtime instances in
	 * the future so it is safer to use this method.
	 * 
	 * @return the Runtime instance for this context.
	 */
	public MoccaRuntime getRuntime();

	/**
	 * Get the command handler registry for this context to register
	 * context-specific handlers.
	 * 
	 * @return A command handler registry instance for this context.
	 */
	public CommandHandlerRegistry getCommandHandlerRegistry();

	/**
	 * Get the state object for this context
	 * 
	 * @return the context's state object
	 * @throws MoccaException
	 *             if the state object cannot be created or retrieved (e.g. some
	 *             other object already uses the key reserved for the state
	 *             object)
	 */
	public State getState() throws MoccaException;

	public void addContextConnectionListener(ContextConnectionListener listener);

	public void removeContextConnectoinListener(
			ContextConnectionListener listener);
}
