/*
 * Copyright 2012 Stefan Haun, Andreas Nürnberger
 * 
 *      Data and Knowledge Engineering Group, 
 * 		Faculty of Computer Science,
 *		Otto-von-Guericke University,
 *		Magdeburg, Germany
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.ovgu.dke.mocca.api.command;

import java.net.URI;

import net.jcip.annotations.ThreadSafe;

/**
 * A registry for {@link CommandHandler}S.
 * 
 * <p>
 * There is one global registry per MOCCA shell containing the known handlers
 * for incoming commands. Additionally, each context may have a distinct
 * registry which overrides settings from the global command registry.
 * </p>
 * 
 * @author Stefan Haun (stefan.haun@ovgu.de)
 * 
 */
@ThreadSafe
public interface CommandHandlerRegistry {
	/**
	 * Register a command handler. If the command is already known, the entry
	 * will be overwritten.
	 * 
	 * @param hnd
	 *            The handler. Command URIs are provided by the handlers.
	 * @throws NullPointerException
	 *             if the handler is null.
	 */
	public void registerCommandHandler(final CommandHandler hnd);

	/**
	 * Remove a command handler.
	 * 
	 * <p>
	 * Please note: If a command handler registers multiple URIs, each of them
	 * needs to be removed separately!
	 * </p>
	 * 
	 * @param command
	 *            The command URI to be removed.
	 * @throws NullPointerException
	 *             if the command argument is null.
	 */
	public void removeCommandHandler(final URI command);

	/**
	 * Get the command handler for a specific URI.
	 * 
	 * @param command
	 *            The command URI.
	 * @return The command handler or null if no handler is registered for the
	 *         command URI.
	 * @throws NullPointerException
	 *             if the command argument is null.
	 */
	public CommandHandler getCommandHandler(final URI command);
}
