/*
 * Copyright 2012 Stefan Haun, Andreas Nürnberger
 * 
 *      Data and Knowledge Engineering Group, 
 * 		Faculty of Computer Science,
 *		Otto-von-Guericke University,
 *		Magdeburg, Germany
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.ovgu.dke.mocca.api.command;

import java.net.URI;
import java.util.Collection;

import de.ovgu.dke.mocca.api.MoccaException;
import de.ovgu.dke.mocca.api.context.Context;

/**
 * <p>
 * Interface for command handlers. Put the business logic of your application
 * here.
 * </p>
 * 
 * <p>
 * The command handler must be state-less and the handle command method may be
 * called in different threads simultaneously! If you need to store information
 * put them in the context.
 * </p>
 * 
 * @author Stefan Haun (stefan.haun@ovgu.de)
 */
public interface CommandHandler {
	/**
	 * Get the list of commands this handler can process.
	 * 
	 * @return A Collection of URIs denoting the commands supported by this
	 *         handler.
	 */
	public Collection<URI> getAvailableCommands();

	/**
	 * <p>
	 * Handle a Mocca command.
	 * </p>
	 * 
	 * <p>
	 * Command handling should never result in an exception from the
	 * business-logic. If anything goes wrong during processing the command
	 * which is not directly related to Mocca handling (which should be almost
	 * everything), send an error message instead!
	 * </p>
	 * 
	 * @param cmd
	 *            The command
	 * @param ctx
	 *            The context in which the command has been received.
	 * @throws MoccaException
	 *             if the MOCCA part of command handling fails. Do not throw
	 *             business-logic exceptions here!
	 */
	public void handleCommand(final Command cmd, final Context ctx)
			throws MoccaException;
}
