/*
 * Copyright 2012 Stefan Haun, Andreas Nürnberger
 * 
 *      Data and Knowledge Engineering Group, 
 * 		Faculty of Computer Science,
 *		Otto-von-Guericke University,
 *		Magdeburg, Germany
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.ovgu.dke.mocca.api;

import java.net.URI;
import java.util.Properties;

import net.jcip.annotations.ThreadSafe;

import de.ovgu.dke.mocca.api.command.Command;
import de.ovgu.dke.mocca.api.command.CommandHandlerRegistry;
import de.ovgu.dke.mocca.api.context.Context;

/**
 * Mocca Runtime environment control methods.
 * 
 * @author Stefan Haun (stefan.haun@ovgu.de)
 */
@ThreadSafe
public interface MoccaRuntime {
	/**
	 * <code>null</code> environment, i.e. no special configuration given.
	 */
	public static final Properties NULL_ENV = null;

	/**
	 * Initialize the Mocca environment
	 * 
	 * @param env
	 *            The environment which is handed to underlying implementations.
	 *            This allows to add special configuration, e.g. for GLUE XMPP.
	 *            May be <code>null</code> if no environment is provided.
	 * @throws MoccaException
	 *             if the initialization fails
	 */
	public void init(final Properties env) throws MoccaException;

	/**
	 * Dispose the Mocca environment
	 * 
	 * @throws MoccaException
	 *             if something goes wrong. However, the environment is most
	 *             likely unusable anyway.
	 */
	public void dispose() throws MoccaException;

	/**
	 * Create a new communication context.
	 * 
	 * @return A context instance in this runtime environment.
	 * 
	 * @throws MoccaException
	 *             if the context cannot be created.
	 */
	public Context createContext() throws MoccaException;

	/**
	 * Create a new command.
	 * 
	 * @param command
	 *            The command URI
	 * @param parameters
	 *            Parameter set matching the command.
	 * @return A command instance.
	 * @throws MoccaException
	 *             if the command cannot be created.
	 */
	public Command createCommand(final URI command, final Properties parameters)
			throws MoccaException;

	/**
	 * Get the global command handler registry. Use to register your global
	 * handlers.
	 * 
	 * @return The instance of the global command handler registry.
	 */
	public CommandHandlerRegistry getCommandHandlerRegistry();
}
