package de.otto.eventsourcing.query;

import de.otto.eventsourcing.event.Key;
import de.otto.eventsourcing.event.Payload;
import de.otto.eventsourcing.monitor.TopicUpdateEvent;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.slf4j.Logger;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.kafka.annotation.KafkaListener;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArrayList;

import static java.util.Collections.unmodifiableCollection;
import static java.util.Optional.ofNullable;
import static org.slf4j.LoggerFactory.getLogger;

public class QueryService<T> {
    private static final Logger LOG = getLogger(QueryService.class);

    private final ConcurrentMap<String, T> state = new ConcurrentHashMap<>(1000);
    private final List<ConsumerRecordCallback> callbacks = new CopyOnWriteArrayList<>();

    private final EventProcessor<T> eventProcessor;
    private final ApplicationEventPublisher eventPublisher;

    public QueryService(final EventProcessor<T> eventProcessor,
                        final ApplicationEventPublisher eventPublisher) {
        this.eventProcessor = eventProcessor;
        this.eventPublisher = eventPublisher;
    }

    public QueryService(final EventProcessor<T> eventProcessor) {
        this.eventProcessor = eventProcessor;
        this.eventPublisher = null;
    }

    public final void addCallback(final ConsumerRecordCallback callback) {
        callbacks.add(callback);
        LOG.trace("Registered Callback #{}", callbacks.size());
    }

    public final void removeCallback(final ConsumerRecordCallback callback) {
        callbacks.remove(callback);
        LOG.trace("Unregistered Callback. #{} remaining.", callbacks.size());
    }

    @KafkaListener(topics = "${eventsourcing.topics.default}")
    public void receive(final ConsumerRecord<Key, Payload> consumerRecord) {
        final Key key = consumerRecord.key();
        if (consumerRecord.value() != null) {
            state.compute(keyForEntity(consumerRecord), (x, existing) -> eventProcessor.process(consumerRecord, ofNullable(existing)));
        } else {
            state.remove(keyForEntity(consumerRecord));
        }
        LOG.info("Received key='{}', payload='{}', partition='{}', offset='{}', event='{}'", key, consumerRecord.value(), consumerRecord.partition(), consumerRecord.offset());
        notifyCallbacks(consumerRecord);
        if (eventPublisher != null) {
            eventPublisher.publishEvent(new TopicUpdateEvent(consumerRecord));
        }
    }

    protected String keyForEntity(ConsumerRecord<Key, Payload> consumerRecord) {
        return consumerRecord.key().getEntityId();
    }

    private void notifyCallbacks(ConsumerRecord<Key, Payload> consumerRecord) {
        this.callbacks.forEach(c -> c.onSuccess(consumerRecord));
    }

    public final T get(final String id) {
        return state.get(id);
    }

    public final Collection<T> getAll() {
        return unmodifiableCollection(state.values());
    }

    public final int size() {
        return state.size();
    }

    /**
     * ONLY for testing purposes!
     */
    public final void deleteAll() {
        state.clear();
    }
}
