package de.otto.eventsourcing.monitor;

import kafka.admin.AdminClient;
import org.apache.kafka.common.TopicPartition;
import org.slf4j.Logger;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.annotation.Scheduled;

import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import static java.lang.Long.MAX_VALUE;
import static java.util.Comparator.naturalOrder;
import static org.slf4j.LoggerFactory.getLogger;
import static scala.collection.JavaConversions.mapAsJavaMap;

public class TopicsMonitor {

    private static final Logger LOG = getLogger(TopicsMonitor.class);

    @SuppressWarnings("MismatchedQueryAndUpdateOfCollection")
    private static final ConcurrentHashMap<Integer, Long> EMPTY_CONCURRENT_MAP = new ConcurrentHashMap<>();
    private static final String CONSUMER_OFFSETS_TOPIC = "__consumer_offsets";

    private final ConcurrentMap<String, ConcurrentMap<Integer, Long>> serverPartitionOffsets = new ConcurrentHashMap<>();
    private final ConcurrentMap<String, ConcurrentMap<Integer, Long>> consumerPartitionOffsets = new ConcurrentHashMap<>();
    private final AdminClient adminClient;
    private final Set<String> groupIds = Collections.synchronizedSet(new HashSet<>());
    private final long acceptedLag;

    public TopicsMonitor(final AdminClient adminClient,
                         final long acceptedLag) {
        LOG.info("Starting TopicsMonitor");
        this.adminClient = adminClient;
        this.acceptedLag = acceptedLag;
    }

    public void registerGroupId(final String groupId) {
        groupIds.add(groupId);
    }

    public void unregisterGroupId(final String groupId) {
        groupIds.remove(groupId);
    }

    @EventListener
    public void onApplicationEvent(final TopicUpdateEvent event) {
        synchronized (consumerPartitionOffsets) {
            consumerPartitionOffsets.putIfAbsent(event.getTopic(), new ConcurrentHashMap<>());
            consumerPartitionOffsets.get(event.getTopic()).put(event.getPartition(), event.getOffset());
        }
    }

    public boolean isUpToDate(final String topic) {
        return lag(topic) <= acceptedLag;
    }

    public long lag(final String topic) {
        if (serverPartitionOffsets.containsKey(topic)) {
            return serverPartitionOffsets.get(topic).keySet()
                    .stream()
                    .map(partition -> {
                        final long serverOffset = serverPartitionOffsets.get(topic).get(partition);
                        final long consumerOffset = consumerPartitionOffsets.getOrDefault(topic, EMPTY_CONCURRENT_MAP).getOrDefault(partition, 0L);
                        return serverOffset - consumerOffset - 1;
                    })
                    .max(naturalOrder())
                    .orElse(MAX_VALUE);
        } else {
            return MAX_VALUE;
        }
    }

    @Scheduled(
                // for some strange reason it is not possible to use default values
                // configured in EventsourcingProperties, so I have to set them here, too.
                // The values here should be consistent with defaults configured in EventsourcingProperties.
                initialDelayString = "${eventsourcing.topics-monitor.initial-delay:500}",
                fixedDelayString = "${eventsourcing.topics-monitor.fetch-delay:5000}"
    )
    public void fetchGroupOffsets() {
        try {
            groupIds.forEach(consumerGroup -> {
                final Map<TopicPartition, Object> topicPartitionMap = mapAsJavaMap(adminClient.listGroupOffsets(consumerGroup));
                topicPartitionMap.keySet().forEach(topicPartition -> {
                    final String topic = topicPartition.topic();
                    if (!topic.equals(CONSUMER_OFFSETS_TOPIC)) {
                        @SuppressWarnings("unchecked") final Long offset = (Long) topicPartitionMap.get(topicPartition);
                        final int partition = topicPartition.partition();
                        serverPartitionOffsets.putIfAbsent(topic, new ConcurrentHashMap<>());
                        serverPartitionOffsets.get(topic).put(partition, offset);
                        LOG.trace("Group={}, Topic={}, Partition={}, Offset={}, Lag={}, isUpToDate={}", consumerGroup, topic, partition, offset, lag(topic), isUpToDate(topic));
                    }
                });
            });
        } catch (final RuntimeException e) {
            LOG.error("Exception caught while fetching group offsets: " + e.getMessage(), e);
        }
    }

    public Set<String> getTopics() {
        return serverPartitionOffsets.keySet();
    }

    public int getNumberOfPartitions(final String topic) {
        return serverPartitionOffsets.getOrDefault(topic, EMPTY_CONCURRENT_MAP).keySet().size();
    }
}
