package de.otto.eventsourcing.event;

import com.fasterxml.jackson.annotation.JsonAnyGetter;
import com.fasterxml.jackson.annotation.JsonAnySetter;
import com.fasterxml.jackson.annotation.JsonIgnore;
import org.hibernate.validator.constraints.NotBlank;
import org.hibernate.validator.constraints.NotEmpty;

import java.util.*;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyMap;

/**
 * The parts of an Event.
 * <p>
 *     Keys may either be simple (a single String) or complex, consisting of multiple
 *     parts. The parts are separated using '/' characters.
 * </p>
 */
public class Key {

    @NotBlank
    private String entityId;
    @NotEmpty
    private SortedMap<String,String> parts = new TreeMap<>();

    Key() {
        entityId = "";
    }

    private Key(final String entityId,
                final String k1, final String v1,
                final String... more) {
        final List<String> keyValues = asList(more);
        if (keyValues.size() % 2 != 0) {
            throw new IllegalArgumentException("number of key-values must be even");
        }
        this.entityId = entityId;
        this.parts.put(k1, v1);
        for (int i = 0; i < keyValues.size() / 2; i++) {
            this.parts.put(keyValues.get(2*i), keyValues.get(2*i+1));
        }
    }

    private Key(final String entityId,
                final Map<String, String> parts) {
        this.entityId = entityId;
        this.parts.putAll(parts);
    }

    public static Key of(final String entityId) {
        return new Key(entityId, emptyMap());
    }

    public static Key of(final String entityId, final String key, final String value) {
        return new Key(entityId, key, value);
    }

    public static Key of(final String entityId,
                         final String k1, final String v1,
                         final String... more) {
        return new Key(entityId, k1, v1, more);
    }

    public static Key of(final String entityId,
                         final Map<String, String> parts) {
        return new Key(entityId, parts);
    }

    @JsonIgnore
    public boolean isComplex() {
        return parts.size() > 0;
    }

    public String getEntityId() {
        return entityId;
    }

    public void setEntityId(final String entityId) {
        this.entityId = entityId;
    }

    @JsonAnyGetter
    public Map<String,String> getParts() {
        return parts;
    }

    @JsonAnySetter
    public void setParts(final String key, final String value) {
        this.parts.put(key, value);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        Key key1 = (Key) o;
        return Objects.equals(entityId, key1.entityId) &&
                Objects.equals(parts, key1.parts);
    }

    @Override
    public int hashCode() {
        return Objects.hash(entityId, parts);
    }

    @Override
    public String toString() {
        return "{" +
                "entityId='" + entityId + '\'' +
                ", parts=" + parts +
                '}';
    }

}
