package de.otto.eventsourcing.configuration;

import com.fasterxml.jackson.databind.ObjectMapper;
import de.otto.eventsourcing.event.Key;
import de.otto.eventsourcing.event.Payload;
import org.apache.kafka.common.serialization.Serializer;
import org.springframework.boot.autoconfigure.kafka.KafkaProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.kafka.core.DefaultKafkaProducerFactory;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.kafka.core.ProducerFactory;
import org.springframework.kafka.support.LoggingProducerListener;
import org.springframework.kafka.support.ProducerListener;
import org.springframework.kafka.support.serializer.JsonSerializer;

@EnableConfigurationProperties(KafkaProperties.class)
@Configuration
public class KafkaProducerConfiguration {

    @Bean
    public ProducerFactory<Key, Payload> eventProducerFactory(final Serializer<Key> keySerializer,
                                                            final Serializer<Payload> payloadSerializer,
                                                            final KafkaProperties kafkaProperties) {
        return new DefaultKafkaProducerFactory<>(
                kafkaProperties.buildProducerProperties(),
                keySerializer,
                payloadSerializer
        );
    }

    @Bean
    public ProducerListener<Key, Payload> eventProducerListener() {
        return new LoggingProducerListener<>();
    }

    @Bean
    public KafkaTemplate<Key, Payload> eventKafkaTemplate(final ProducerFactory<Key,Payload> eventProducerFactory,
                                                        final ProducerListener<Key, Payload> eventProducerListener,
                                                        final KafkaProperties kafkaProperties) {
        KafkaTemplate<Key, Payload> kafkaTemplate =  new KafkaTemplate<>(eventProducerFactory);
        kafkaTemplate.setDefaultTopic(kafkaProperties.getTemplate().getDefaultTopic());
        kafkaTemplate.setProducerListener(eventProducerListener);
        return kafkaTemplate;
    }

    @Bean
    public Serializer<Key> keySerializer(final ObjectMapper objectMapper) {
        return new JsonSerializer<>(objectMapper);
    }

    @Bean
    public Serializer<Payload> payloadSerializer(final ObjectMapper objectMapper) {
        return new JsonSerializer<>(objectMapper);
    }

}
