package de.otto.eventsourcing.configuration;

import com.fasterxml.jackson.databind.ObjectMapper;
import de.otto.eventsourcing.event.Key;
import de.otto.eventsourcing.event.Payload;
import de.otto.eventsourcing.monitor.TopicsMonitor;
import org.apache.kafka.common.serialization.Deserializer;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.kafka.KafkaProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.kafka.config.ConcurrentKafkaListenerContainerFactory;
import org.springframework.kafka.core.ConsumerFactory;
import org.springframework.kafka.support.serializer.JsonDeserializer;

import java.util.Optional;

@EnableConfigurationProperties(KafkaProperties.class)
@Configuration
public class KafkaConsumerConfiguration {

    @Autowired(required = false)
    private TopicsMonitor topicsMonitor;

    @Bean
    public ConcurrentKafkaListenerContainerFactory<Key, Payload> kafkaListenerContainerFactory(final ConsumerFactory<Key,Payload> eventConsumerFactory) {
        final ConcurrentKafkaListenerContainerFactory<Key, Payload> factory = new ConcurrentKafkaListenerContainerFactory<>();
        factory.setConsumerFactory(eventConsumerFactory);
        return factory;
    }

    @Bean
    public ConsumerFactory<Key, Payload> eventConsumerFactory(final KafkaProperties kafkaProperties,
                                                            final Deserializer<Key> keyDeserializer,
                                                            final Deserializer<Payload> payloadDeserializer) {
        return new EventsourcingConsumerFactory(
                kafkaProperties.buildConsumerProperties(),
                keyDeserializer,
                payloadDeserializer,
                Optional.ofNullable(topicsMonitor));
    }

    @Bean
    public Deserializer<Key> keyDeserializer(final ObjectMapper objectMapper) {
        return new JsonDeserializer<>(Key.class, objectMapper);
    }

    @Bean
    public Deserializer<Payload> payloadDeserializer() {
        return new JsonDeserializer<>(Payload.class);
    }
}
