package de.otto.eventsourcing.configuration;

import de.otto.eventsourcing.event.Key;
import de.otto.eventsourcing.event.Payload;
import de.otto.eventsourcing.monitor.TopicsMonitor;
import de.otto.eventsourcing.query.EventsourcingConsumer;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.common.serialization.Deserializer;
import org.springframework.kafka.core.ConsumerFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;

import static java.util.Collections.unmodifiableMap;

/**
 * A ConsumerFactory that is creating EventsourcingConsumers having a unique group-id for every consumer.
 * <p>
 * Multiple Consumers for the same topic must have different groups, so every consumer is
 * able to process all events.
 * </p>
 */
public class EventsourcingConsumerFactory  implements ConsumerFactory<Key, Payload> {

    private final Map<String, Object> configs;

    private Deserializer<Key> keyDeserializer;

    private Deserializer<Payload> valueDeserializer;

    private Optional<TopicsMonitor> topicsMonitor;

    public EventsourcingConsumerFactory(Map<String, Object> configs,
                                        Deserializer<Key> keyDeserializer,
                                        Deserializer<Payload> valueDeserializer,
                                        Optional<TopicsMonitor> topicsMonitor) {
        this.configs = new HashMap<>(configs);
        this.keyDeserializer = keyDeserializer;
        this.valueDeserializer = valueDeserializer;
        this.topicsMonitor = topicsMonitor;
    }

    /**
     * Return an unmodifiable reference to the configuration map for this factory.
     * Useful for cloning to make a similar factory.
     * @return the configs.
     * @since 1.0.6
     */
    public Map<String, Object> getConfigurationProperties() {
        return unmodifiableMap(this.configs);
    }

    @Override
    public EventsourcingConsumer createConsumer() {
        return createKafkaConsumer();
    }

    protected EventsourcingConsumer createKafkaConsumer() {
        return createKafkaConsumer(this.configs);
    }

    protected EventsourcingConsumer createKafkaConsumer(String clientIdSuffix) {
        if (!this.configs.containsKey(ConsumerConfig.CLIENT_ID_CONFIG) || clientIdSuffix == null) {
            return createKafkaConsumer();
        }
        else {
            Map<String, Object> modifiedClientIdConfigs = new HashMap<>(this.configs);
            modifiedClientIdConfigs.put(ConsumerConfig.CLIENT_ID_CONFIG,
                    modifiedClientIdConfigs.get(ConsumerConfig.CLIENT_ID_CONFIG) + clientIdSuffix);
            return createKafkaConsumer(modifiedClientIdConfigs);
        }
    }

    protected EventsourcingConsumer createKafkaConsumer(final Map<String, Object> configs) {
        final String groupId = UUID.randomUUID().toString();
        final Map<String,Object> configsWithGroup = new HashMap<>(configs);
        configsWithGroup.put(ConsumerConfig.GROUP_ID_CONFIG, groupId);
        topicsMonitor.ifPresent(monitor -> monitor.registerGroupId(groupId));
        return new EventsourcingConsumer(configsWithGroup, keyDeserializer, this.valueDeserializer);
    }

    @Override
    public boolean isAutoCommit() {
        Object auto = this.configs.get(ConsumerConfig.ENABLE_AUTO_COMMIT_CONFIG);
        return auto instanceof Boolean ? (Boolean) auto
                : auto instanceof String ? Boolean.valueOf((String) auto) : true;
    }
}
