package de.otto.eventsourcing.command;

import de.otto.eventsourcing.event.Key;
import de.otto.eventsourcing.event.Payload;
import org.slf4j.Logger;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.kafka.support.KafkaNull;

import java.util.List;

import static org.slf4j.LoggerFactory.getLogger;
import static org.springframework.kafka.support.KafkaHeaders.MESSAGE_KEY;
import static org.springframework.kafka.support.KafkaHeaders.TOPIC;
import static org.springframework.messaging.support.MessageBuilder.withPayload;

public class CommandService<T> {
    private static final Logger LOG = getLogger(CommandService.class);

    private final KafkaTemplate<Key, Payload> kafka;
    private final CommandFactory<T> commandFactory;

    public CommandService(final KafkaTemplate<Key, Payload> eventKafkaTemplate,
                          final CommandFactory<T> commandFactory) {
        this.kafka = eventKafkaTemplate;
        this.commandFactory = commandFactory;
    }

    protected void send(final List<Command> commands) {
        commands.forEach(this::send);
    }

    protected void send(final Command command) {
        LOG.info("Sending event='{}'", command);
        if (command.getPayload() == null) {
            kafka.send(withPayload(KafkaNull.INSTANCE)
                    .setHeader(MESSAGE_KEY, command.getKey())
                    .setHeader(TOPIC, command.getTopic())
                    .build()
            );
        } else {
            kafka.send(withPayload(command.getPayload())
                    .setHeader(MESSAGE_KEY, command.getKey())
                    .setHeader(TOPIC, command.getTopic())
                    .build());
        }
    }

    public void put(final T entity) {
        send(commandFactory.putEventsFor(entity));
    }

    public void patch(final T entity) {
        send(commandFactory.patchEventsFor(entity));
    }

    public void delete(final String entityId) {
        send(commandFactory.deleteEventsFor(entityId));
    }

}
