package de.otto.eventsourcing.query;

import de.otto.eventsourcing.event.Event;
import org.slf4j.Logger;
import org.springframework.util.concurrent.ListenableFutureCallback;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArrayList;

import static java.util.Collections.unmodifiableCollection;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static org.slf4j.LoggerFactory.getLogger;

public class QueryService<T> {
    private static final Logger LOG = getLogger(QueryService.class);

    private final ConcurrentMap<String, T> comboState = new ConcurrentHashMap<>(1000);
    private final List<ListenableFutureCallback<Event>> callbacks = new CopyOnWriteArrayList<>();

    private final EventProcessor<T> eventProcessor;

    protected QueryService(final EventProcessor<T> eventProcessor) {
        this.eventProcessor = eventProcessor;
    }

    public final void addCallback(final ListenableFutureCallback<Event> callback) {
        callbacks.add(callback);
        LOG.trace("Registered Callback #{}", callbacks.size());
    }

    public final void removeCallback(final ListenableFutureCallback<Event> callback) {
        callbacks.remove(callback);
        LOG.trace("Unregistered Callback. #{} remaining.", callbacks.size());
    }

    public void receive(final Event event) {
        switch (event.getType()) {
            case CREATE:
                comboState.computeIfAbsent(event.getKey().getEntityId(), (x) -> eventProcessor.process(event, empty()));
                break;
            case PUT:
                comboState.compute(event.getKey().getEntityId(), (x, existing) -> eventProcessor.process(event, ofNullable(existing)));
                break;
            case PATCH:
                comboState.computeIfPresent(event.getKey().getEntityId(), (x, existing) -> eventProcessor.process(event, of(existing)));
                break;
            case DELETE:
                comboState.remove(event.getKey().getEntityId());
                break;
        }
        LOG.trace("Received event='{}'", event);
        this.callbacks.forEach(c -> {
            c.onSuccess(event);
        });
    }

    public final T get(final String id) {
        return comboState.get(id);
    }

    public final Collection<T> getAll() {
        return unmodifiableCollection(comboState.values());
    }

    public final int size() {
        return comboState.size();
    }

    /**
     * ONLY for testing purposes!
     */
    public final void deleteAll() {
        comboState.clear();
    }
}
