package de.otto.eventsourcing.command;

import de.otto.eventsourcing.event.Event;
import de.otto.eventsourcing.event.Key;
import org.slf4j.Logger;
import org.springframework.boot.autoconfigure.kafka.KafkaProperties;
import org.springframework.kafka.core.KafkaTemplate;

import java.util.List;

import static org.slf4j.LoggerFactory.getLogger;
import static org.springframework.kafka.support.KafkaHeaders.MESSAGE_KEY;
import static org.springframework.kafka.support.KafkaHeaders.TOPIC;
import static org.springframework.messaging.support.MessageBuilder.withPayload;

public class CommandService<T> {
    private static final Logger LOG = getLogger(CommandService.class);

    private final KafkaTemplate<Key, Event> kafka;
    private final EventFactory<T> eventFactory;
    private final String topic;

    public CommandService(final KafkaTemplate<Key, Event> eventKafkaTemplate,
                          final EventFactory<T> eventFactory,
                          final String topic) {
        this.kafka = eventKafkaTemplate;
        this.eventFactory = eventFactory;
        this.topic = topic;
    }

    public CommandService(final KafkaTemplate<Key, Event> eventKafkaTemplate,
                          final EventFactory<T> eventFactory,
                          final KafkaProperties kafkaProperties) {
        this(eventKafkaTemplate, eventFactory, kafkaProperties.getTemplate().getDefaultTopic());
    }

    protected void send(final List<Event> events) {
        events.forEach(this::send);
    }

    protected void send(final Event event) {
        LOG.trace("Sending event='{}' to topic='{}'", event, topic);
        kafka.send(withPayload(event)
                .setHeader(MESSAGE_KEY, event.getKey())
                .setHeader(TOPIC, topic)
                .build());
    }

    public void create(final T entity) {
        send(eventFactory.createEventsFor(entity));
    }

    public void put(final T entity) {
        send(eventFactory.putEventsFor(entity));
    }

    public void patch(final T entity) {
        send(eventFactory.patchEventsFor(entity));
    }

    public void delete(final String entityId) {
        send(eventFactory.deleteEventsFor(entityId));
    }

}
