package de.osshangar.plugin.classloader;

import lombok.Getter;
import lombok.NonNull;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

/**
 * Class loader that stores byte code of known classes in RAM
 * */
public class PluginClassLoader extends ClassLoader {
    // contains all class files that are known to the class loader with their name
    @Getter
    private final Map<String, byte[]> byteCodes;

    private final Map<String, byte[]> resources;

    /**
     * Creates the class loader
     * @param classesFiles Map of full qualified class names as keys and the byte code of the classes as values
     */
    public PluginClassLoader(@NonNull Map<String, byte[]> classesFiles, @NonNull Map<String, byte[]> resources) {
        super(PluginClassLoader.class.getClassLoader());
        this.byteCodes = classesFiles;
        this.resources = resources;
    }

    @Override
    public Class<?> findClass(@NonNull String fullQualifiedClassName) throws ClassNotFoundException {
        if (!byteCodes.containsKey(fullQualifiedClassName)){
            throw new ClassNotFoundException(String.format("Did not find class '%s'", fullQualifiedClassName));
        }
        return defineClass(fullQualifiedClassName, byteCodes.get(fullQualifiedClassName), 0, byteCodes.get(fullQualifiedClassName).length);
    }

    @Override
    public Class<?> loadClass(@NonNull String fullQualifiedClassName) throws ClassNotFoundException {
        if (byteCodes.containsKey(fullQualifiedClassName)){
            return findClass(fullQualifiedClassName);
        }
        ClassLoader defaultLoader = Thread.currentThread().getContextClassLoader();
        return defaultLoader.loadClass(fullQualifiedClassName);
    }

    @Override
    public InputStream getResourceAsStream(String name) {
        File file = new File(name);
        if (!file.exists()){
            // Resource does not exist in the specified path, try to find it in the loaded byte code of the plugin JAR
            if (resources.containsKey(name)){
                return new ByteArrayInputStream(resources.get(name));
            }
        }
        try (InputStream inputStream = super.getResourceAsStream(name)) {
            return inputStream;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
