package de.osshangar.plugin;

import de.osshangar.plugin.classloader.PluginClassLoader;
import de.osshangar.plugin.content.Inspector;
import de.osshangar.plugin.exception.FileFormatException;
import lombok.NonNull;
import lombok.RequiredArgsConstructor;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.jar.JarInputStream;

/**
 * This class loads the plugins
 * @param <T> type of the object created
 * @param <C> the interface class that the plugin has to implement
 */
@RequiredArgsConstructor
public class Plugin<T, C extends Class<T>> {
    @NonNull
    private final C interfaceClass;

    /**
     * This loads the plugin from the provided JarInputStream.
     * @param jarInputStream The input stream to a jar package to load the plugin from
     * @return The instantiated plugin instance
     */
    public T load(JarInputStream jarInputStream) throws FileFormatException, IOException, ClassNotFoundException, InvocationTargetException, NoSuchMethodException, InstantiationException, IllegalAccessException {
        return load(jarInputStream, null);
    }

    public T load(JarInputStream jarInputStream, Arguments constructorArguments) throws FileFormatException, IOException, ClassNotFoundException, InvocationTargetException, NoSuchMethodException, InstantiationException, IllegalAccessException {
        Inspector inspector = Inspector.inspect(jarInputStream, interfaceClass);
        PluginClassLoader classLoader = new PluginClassLoader(inspector.getClassFiles());
        return createInstance(classLoader, inspector, constructorArguments);
    }

    private T createInstance(PluginClassLoader classLoader, Inspector inspector, Arguments constructorArguments) throws ClassNotFoundException, NoSuchMethodException, InvocationTargetException, InstantiationException, IllegalAccessException {
        Class<?> pluginClass = classLoader.findClass(inspector.getPluginClassName());
        if (!interfaceClass.isAssignableFrom(pluginClass)){
            throw new ClassCastException(String.format("The compiled class is no implementation or subclass of %s", interfaceClass.getCanonicalName()));
        }

        Class<?>[] signature = new Class[]{};
        Object[] arguments = new Object[]{};
        if (constructorArguments != null){
            signature = constructorArguments.getConstructor();
            arguments = constructorArguments.getArguments();
        }

        Constructor<?> constructor = pluginClass.getConstructor(signature);
        //noinspection unchecked
        return (T) constructor.newInstance(arguments);
    }
}
