package de.osci.osci12.messageparts;

import de.osci.helper.CustomHeaderExtention;
import eu.osci.ws._2014._10.transport.MessageMetaData;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;


/**
 * Diese CustomHeaderExtention bietet die Möglichkeit, ein MessageMetaData-Objekt so aufzubereiten, dass es in
 * einer OSCI-Nachricht als korrekter Custom SOAP Header gesetzt werden kann, z.B. mit dem Aufruf
 * {@link de.osci.osci12.messagetypes.StoreDelivery#addCustomHeaderExtention(CustomHeaderExtention)}.
 */
public class MessageMetaDataCustomHeader extends CustomHeaderExtention
{

  private static final String MESSAGE_META_DATA_ID = "MetaData";


  private MessageMetaData messageMetaData;

  private static Logger log = LoggerFactory.getLogger(MessageMetaDataCustomHeader.class);

  private static JAXBContext jaxbContext;

  private static final XMLInputFactory XML_INPUT_FACTORY = XMLInputFactory.newFactory();

  static
  {
    try
    {
      jaxbContext = JAXBContext.newInstance(eu.osci.ws._2014._10.transport.ObjectFactory.class);
      // XML Factory gegen XXE absichern
      XML_INPUT_FACTORY.setProperty(XMLInputFactory.IS_SUPPORTING_EXTERNAL_ENTITIES, false);
      XML_INPUT_FACTORY.setProperty(XMLInputFactory.SUPPORT_DTD, false);
    }
    catch (final JAXBException e)
    {
      log.error("Cannot create JAXB Context for MessageMetaData, handling MessageMetaData will be impossible",
                e);
    }
  }

  /**
   * Erzeugt ein neues MessageMetaDataCustomHeader-Objekt mit einem {@link MessageMetaData}
   *
   * @param messageMetaData
   */
  public MessageMetaDataCustomHeader(MessageMetaData messageMetaData)
  {
    this.messageMetaData = messageMetaData;
  }

  public MessageMetaDataCustomHeader()
  {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getOutgoingHeader()
  {
    String mmdOsciString = "";

    String mmdStringWithId = addIdToCustomHeader(getMessageMetaDataAsString(messageMetaData),
                                                 MESSAGE_META_DATA_ID);

    mmdOsciString = makeOSCICustomHeader(mmdStringWithId);

    if (log.isDebugEnabled())
    {
      log.debug("get MessageMetaData: " + mmdOsciString);
    }
    return mmdOsciString;
  }

  public MessageMetaData getMessageMetaData()
  {
    return messageMetaData;
  }


  /**
   * Erzeuge eine Zeichenkette aus einem unveränderten {@link MessageMetaData}-Objekt.
   *
   * @param mmd
   * @return
   */
  private static String getMessageMetaDataAsString(final MessageMetaData mmd)
  {
    String mmdString = "";

    try
    {
      final ByteArrayOutputStream streamMMD = new ByteArrayOutputStream();
      jaxbContext.createMarshaller().marshal(mmd, streamMMD);

      mmdString = new String(streamMMD.toByteArray(), StandardCharsets.UTF_8);
    }
    catch (final Exception ex)
    {
      log.warn("Could not add messageMetaData!", ex);
    }
    return mmdString;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getCustomHeaderRefID()
  {
    return MESSAGE_META_DATA_ID;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CustomHeaderExtention parseIncomingHeader(final String customHeaderString) throws IOException
  {
    try (final Reader in = new StringReader(customHeaderString))
    {
      messageMetaData = (MessageMetaData)jaxbContext.createUnmarshaller()
                                                    .unmarshal(XML_INPUT_FACTORY.createXMLStreamReader(in));
      return this;
    }
    catch (final IOException | JAXBException | XMLStreamException ex)
    {
      throw new IOException(ex);
    }
  }
}
