/* BSD 2-Clause License:
 * Copyright (c) 2009 - 2017
 * Software Technology Group
 * Department of Computer Science
 * Technische Universität Darmstadt
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package org.opalj
package br

import java.lang.ref.WeakReference
import java.util.WeakHashMap
import java.util.concurrent.atomic.AtomicInteger
import java.util.concurrent.locks.ReentrantReadWriteLock

import scala.annotation.tailrec
import scala.math.Ordered
import scala.collection.SortedSet

import org.opalj.collection.UID
import org.opalj.collection.immutable.UIDSet
import org.opalj.collection.immutable.UIDSet2

/**
 * Represents a JVM type.
 *
 * Programmatically, we distinguish three major kinds of types:
 *  - base types/primitive types
 *  - reference types
 *  - the type void.
 *
 * ==General Information==
 * ''From the JVM specification''
 *
 * There are three kinds of reference types: class types, array types, and interface
 * types. Their values are references to dynamically created class instances, arrays,
 * or class instances or arrays that implement interfaces, respectively.
 *
 * A reference value may also be the special null reference, a reference to no object,
 * which will be denoted here by null. The null reference initially has no runtime type,
 * but may be cast to any type. The default value of a reference type is null.
 * The Java virtual machine specification does not mandate a concrete value encoding null.
 *
 * ==Comparing Types/Performance==
 * Given that the comparison of types is a standard operation in static analysis that
 * is usually done over and over again great care was taken to enable an efficient
 * comparison of types. It is - '''without exception''' - always possible to compare
 * types using reference equality (i.e., the `eq`/`ne` operators). For each type there
 * will always be at most one object that represents that specific type.
 *
 * Additionally, a stable order is defined between types that is based on a type's
 * kind and the unique id of the types in case of reference types.
 * The order is:
 * ''void type &lt; primitive types &lt; array types &lt; class/interface types''
 *
 * @author Michael Eichberg
 * @author Andre Pacak
 */
sealed abstract class Type extends UID with Ordered[Type] {

    /**
     * Returns `true` if this type can be used by fields. Returns `true` unless
     * this type represents `void`.
     */
    def isFieldType: Boolean = false

    /**
     * Returns `true` if this type represents `void`; `false` otherwise.
     */
    def isVoidType: Boolean = false

    /**
     * Returns `true` if this type is a base type (also called primitive type).
     */
    def isBaseType: Boolean = false

    /**
     * Returns `true` if this type is the primitive type `byte`.
     */
    def isByteType: Boolean = false

    /**
     * Returns `true` if this type is the primitive type `char` (Range: [0..65535].
     */
    def isCharType: Boolean = false

    /**
     * Returns `true` if this type is the primitive type `short`.
     */
    def isShortType: Boolean = false

    /**
     * Returns `true` if this type is the primitive type `int`.
     */
    def isIntegerType: Boolean = false

    /**
     * Returns `true` if this type is the primitive type `long`.
     */
    def isLongType: Boolean = false

    /**
     * Returns `true` if this type is the primitive type `float`.
     */
    def isFloatType: Boolean = false

    /**
     * Returns `true` if this type is the primitive type `double`.
     */
    def isDoubleType: Boolean = false

    /**
     * Returns `true` if this type is the primitive type `boolean`.
     */
    def isBooleanType: Boolean = false

    /**
     * Returns `true` if this type is a reference type; that is, an array type or an
     * object type (class/interface type).
     *
     * @note
     * In general, we can distinguish the following three categories of types:
     *  - base types,
     *  - reference types,
     *  - the type void.
     */
    def isReferenceType: Boolean = false
    def isArrayType: Boolean = false
    def isObjectType: Boolean = false
    def isNumericType: Boolean = false
    def isIntLikeType: Boolean = false

    /**
     * The computational type of values of this type.
     */
    @throws[UnsupportedOperationException](
        "if this type has no associated computational type(i.e., if this type represents void)"
    )
    def computationalType: ComputationalType

    /**
     * The number of operand stack slots/registers required to store
     * a single value of this type. In case of `VoidType` `0` is returned.
     */
    def operandSize: Int = computationalType.operandSize.toInt

    @throws[ClassCastException]("if this type is not a reference type")
    def asReferenceType: ReferenceType = {
        throw new ClassCastException(this.toJava+" cannot be cast to a ReferenceType")
    }

    @throws[ClassCastException]("if this type is not an array type")
    def asArrayType: ArrayType = {
        throw new ClassCastException(this.toJava+" cannot be cast to an ArrayType")
    }

    @throws[ClassCastException]("if this type is not an object type")
    def asObjectType: ObjectType = {
        throw new ClassCastException(this.toJava+" cannot be cast to an ObjectType")
    }

    @throws[ClassCastException]("if this type is not a base type")
    def asBaseType: BaseType = {
        throw new ClassCastException(
            "a "+this.getClass.getSimpleName+" cannot be cast to a BaseType"
        )
    }

    @throws[ClassCastException]("if this type is not a field type")
    def asFieldType: FieldType = {
        val message = s"a ${this.getClass.getSimpleName} cannot be cast to a FieldType"
        throw new ClassCastException(message)
    }

    @throws[ClassCastException]("if this is not a numeric type")
    def asNumericType: NumericType = {
        throw new ClassCastException(
            "a "+this.getClass.getSimpleName+" cannot be cast to a NumericType"
        )
    }

    @throws[ClassCastException]("if this is not a numeric type")
    def asIntLikeType: IntLikeType = {
        throw new ClassCastException(
            "a "+this.getClass.getSimpleName+" cannot be cast to an IntLikeType"
        )
    }

    /**
     * A String representation of this type as it would be used in Java source code.
     */
    def toJava: String

    /**
     * Returns the binary name of this type as used by the Java runtime. Basically
     * returns the same name as produced by `Class.getName`.
     */
    @throws[UnsupportedOperationException](
        "if this type has not binary name(i.e., if this type represents void)"
    )
    def toBinaryJavaName: String

    /**
     * Returns the representation of this type as used by the JVM in, for example,
     * method descriptors or signatures.
     */
    def toJVMTypeName: String

    /**
     * Returns the Java class object representing this type.
     *
     * '''This is generally only useful in very special cases and – to be meaningful at
     * all – it is necessary that the class path used for running the static analysis also
     * contains the classes that are analyzed. This is (often) only the case for the
     * JDK. '''
     *
     * However, one example where this is useful is the creation of a real object of
     * a specific type and to use that object when a method is called on that object.
     * This avoids the reimplementation of the respective logic as part of the analysis.
     * For example, if you want to get the `String` that is created by a specific
     * `StringBuffer` it is possible to implement the API of StringBuffer as part of
     * your analysis or (probably more efficient) to just create an instance of a
     * `StringBuffer` object and to redirect every call to the real object. In this case
     * only some general logic is required to redirect calls and to convert the values
     * between the representation used by the analysis and the representation required
     * by the called method.
     */
    def toJavaClass: java.lang.Class[_]

    /**
     * The unique id of this type. Types are associated with globally unique ids to
     * make it easy to define a global order.
     */
    def id: Int

    /**
     * Compares this type with the given type.
     *
     * Comparison of types is implemented by comparing the associated ids. I.e.,
     * the result of the comparison of two types is '''not stable''' across multiple
     * runs of OPAL.
     */
    override def compare(that: Type): Int = {
        if (this eq that)
            0
        else if (this.id < that.id)
            -1
        else
            1
    }

    override def <(other: Type): Boolean = this.id < other.id
    override def >(other: Type): Boolean = this.id > other.id
    override def >=(other: Type): Boolean = this.id >= other.id
    override def <=(other: Type): Boolean = this.id <= other.id

}

object Type {

    def apply(clazz: Class[_]): Type = {
        if (clazz.isPrimitive) {
            clazz match {
                case java.lang.Boolean.TYPE   ⇒ BooleanType
                case java.lang.Byte.TYPE      ⇒ ByteType
                case java.lang.Character.TYPE ⇒ CharType
                case java.lang.Short.TYPE     ⇒ ShortType
                case java.lang.Integer.TYPE   ⇒ IntegerType
                case java.lang.Long.TYPE      ⇒ LongType
                case java.lang.Float.TYPE     ⇒ FloatType
                case java.lang.Double.TYPE    ⇒ DoubleType
                case java.lang.Void.TYPE      ⇒ VoidType
                case _ ⇒
                    throw new UnknownError(s"unknown primitive type $clazz")
            }
        } else {
            ReferenceType(clazz.getName)
        }
    }
}

object ReturnType {

    def apply(rt: String): Type = if (rt.charAt(0) == 'V') VoidType else FieldType(rt)

}

/**
 * Represents the Java type/keyword `void`.
 *
 * @author Michael Eichberg
 */
sealed abstract class VoidType private () extends Type with ReturnTypeSignature {

    final override def toJVMSignature: String = toJVMTypeName

    final val id = Int.MinValue

    final override def isVoidType: Boolean = true

    final override def computationalType: ComputationalType =
        throw new UnsupportedOperationException("void does not have a computational type")

    final override def operandSize: Int = 0

    final override def accept[T](sv: SignatureVisitor[T]): T = sv.visit(this)

    override def toJava: String = "void"

    override def toBinaryJavaName: String =
        throw new UnsupportedOperationException("void does not have a binary name")

    override def toJVMTypeName: String = "V"

    override def toJavaClass: java.lang.Class[_] = java.lang.Void.TYPE

    override def toString(): String = "VoidType"

}
case object VoidType extends VoidType

/**
 * Supertype of all types except [[VoidType]].
 *
 * @author Michael Eichberg
 */
sealed abstract class FieldType extends Type {

    final override def isFieldType: Boolean = true

    final override def asFieldType: this.type = this

    /**
     * Returns the sequence of instructions that adapts values of `this` type to values
     * of the target type.
     *
     * This method supports the following kind of adaptations:
     *  - boxing
     *  - unboxing
     */
    @throws[IllegalArgumentException]("if a(n) (un)boxing to the targetType is not possible")
    def adapt[T](targetType: Type)(implicit typeConversionFactory: TypeConversionFactory[T]): T
}
/**
 * Factory to parse field type (descriptors) to get field type objects.
 */
object FieldType {

    @throws[IllegalArgumentException](
        "if the given string is not a valid field type descriptor"
    )
    def apply(ft: String): FieldType = {
        (ft.charAt(0): @scala.annotation.switch) match {
            case 'B' ⇒ ByteType
            case 'C' ⇒ CharType
            case 'D' ⇒ DoubleType
            case 'F' ⇒ FloatType
            case 'I' ⇒ IntegerType
            case 'J' ⇒ LongType
            case 'S' ⇒ ShortType
            case 'Z' ⇒ BooleanType
            case 'L' ⇒ ObjectType(ft.substring(1, ft.length - 1))
            case '[' ⇒ ArrayType(FieldType(ft.substring(1)))
            case _   ⇒ throw new IllegalArgumentException(ft+" is not a valid field type descriptor")
        }
    }
}

sealed abstract class ReferenceType extends FieldType {

    final override def isReferenceType: Boolean = true

    final override def asReferenceType: ReferenceType = this

    final override def computationalType = ComputationalTypeReference

    /**
     * Each reference type is associated with a unique id. Object types get ids &gt;= 0
     * and array types get ids &lt; 0.
     */
    def id: Int

}
/**
 * Factory to create instances of `ReferenceType`.
 */
object ReferenceType {

    /**
     * Creates a representation of the described [[ReferenceType]].
     *
     * @param   rt A string as passed to `java.lang.Class.forName(...)` but in binary notation.
     *          Examples:
     *          {{{
     *          "[B" // in case of an array of Booleans
     *          "java/lang/Object" // for the class type java.lang.Object
     *          "[Ljava/lang/Object;" // for the array of java.lang.Object
     *          }}}
     */
    @throws[IllegalArgumentException](
        "if the given string is not a valid reference type descriptor"
    )
    def apply(rt: String): ReferenceType = {
        if (rt.charAt(0) == '[')
            ArrayType(FieldType(rt.substring(1)))
        else
            ObjectType(rt)
    }
}

sealed abstract class BaseType extends FieldType with TypeSignature {

    final override def isBaseType: Boolean = true

    final override def asBaseType: this.type = this

    final override def toJVMSignature: String = toJVMTypeName

    /**
     * The atype value of the base type. The atype value uniquely identifies a base
     * type and is used primarily by the [instruction.NEWARRAY] instruction.
     */
    def atype: Int

    val WrapperType: ObjectType

    def boxValue[T](implicit typeConversionFactory: TypeConversionFactory[T]): T

    final override def adapt[T](
        targetType: Type
    )(
        implicit
        typeConversionFactory: TypeConversionFactory[T]
    ): T = {
        if ((targetType eq WrapperType) || (targetType eq ObjectType.Object)) {
            boxValue
        } else {
            val message = s"adaptation of ${this.toJava} to $targetType is not supported"
            throw new IllegalArgumentException(message)
        }
    }

}

/**
 * Common constants related to base types (aka. primitive types).
 */
object BaseType {

    implicit val BaseTypeOrdering =
        new Ordering[BaseType] {
            def compare(a: BaseType, b: BaseType): Int = a.compare(b)
        }

    /**
     * The set of [BaseType]s sorted by the type's id.
     */
    final val baseTypes: SortedSet[BaseType] =
        SortedSet[BaseType](
            BooleanType,
            ByteType, CharType, ShortType, IntegerType, // <= "IntLike" values
            LongType,
            FloatType,
            DoubleType
        )
}

sealed abstract class NumericType protected () extends BaseType {

    /**
     * Returns the instruction sequence that can convert a value of the current
     * type to `targetType`.
     *
     * For primitive values the appropriate instructions that perform the necessary
     * widening/narrowing are returned. If this type
     * is a primitive type and the target type is a wrapper type, then the object of
     * the corresponding wrapper type is created and returned.
     *
     * @note The functionality implemented here, basically implements the logic
     *      for handling boxing and unboxing operations.
     */
    def convertTo[T](
        targetType: NumericType
    )(
        implicit
        typeConversionFactory: TypeConversionFactory[T]
    ): T

    /**
     * Determines if the range of values captured by `this` type is a '''strict'''
     * superset of the range of values captured by values of type `targetType`. Here,
     * strict superset means that – except of rounding issues – the value is conceptually
     * representable by `this` type. For example, a conversion from a `long` value to a
     * `double` value may loose some precision related to the least significant bits,
     * but the value is still representable.
     *
     * In general, the result of `isWiderThan` is comparable to the result of determing
     * if a conversion of a value of this type to the given type is an explicit/implicit
     * widening conversion.
     *
     * @example
     * {{{
     * assert(IntegerType.isWiderThan(IntegerType) == false)
     * assert(IntegerType.isWiderThan(LongType) == false)
     * assert(IntegerType.isWiderThan(ByteType) == true)
     * assert(LongType.isWiderThan(FloatType) == false)
     * assert(ByteType.isWiderThan(CharType) == false)
     * assert(LongType.isWiderThan(ShortType) == true)
     * }}}
     */
    def isWiderThan(targetType: NumericType): Boolean

    override def isNumericType = true

    override def asNumericType: this.type = this

}

/**
 * An IntLikeType is every type (byte, char, short and int) that uses a primtive int
 * to store the current value and which has explicit support in the JVM.
 *
 * @note `Boolean` values are (at least conceptually) also stored in ints. However, the
 *      JVM has basically no explicit support for booleans (e.g., a conversion of an int
 *      value to a boolean is not directly supported).
 */
sealed abstract class IntLikeType protected () extends NumericType {

    override def isIntLikeType: Boolean = true

    override def asIntLikeType: this.type = this

}

sealed abstract class ByteType private () extends IntLikeType {

    final val atype = 8

    final val id = Int.MinValue + atype

    final val WrapperType = ObjectType.Byte

    final override def isByteType: Boolean = true

    final override def computationalType: ComputationalType = ComputationalTypeInt

    def accept[T](v: SignatureVisitor[T]): T = v.visit(this)

    def toJava: String = "byte"

    override def toBinaryJavaName: String = "B"

    override def toJVMTypeName: String = "B"

    override def toJavaClass: java.lang.Class[_] = java.lang.Byte.TYPE

    override def toString() = "ByteType"

    override def isWiderThan(targetType: NumericType): Boolean = false

    override def convertTo[T](
        targetType: NumericType
    )(
        implicit
        typeConversionFactory: TypeConversionFactory[T]
    ): T = {
        import typeConversionFactory._
        (targetType.id: @scala.annotation.switch) match {
            case ByteType.id |
                ShortType.id |
                IntegerType.id ⇒ NoConversion
            case CharType.id   ⇒ IntToChar
            case LongType.id   ⇒ IntToLong
            case FloatType.id  ⇒ IntToFloat
            case DoubleType.id ⇒ IntToDouble
        }
    }

    override def boxValue[T](implicit typeConversionFactory: TypeConversionFactory[T]): T = {
        typeConversionFactory.PrimitiveByteToLangByte
    }

}
case object ByteType extends ByteType

sealed abstract class CharType private () extends IntLikeType {

    final val atype = 5

    final val id = Int.MinValue + atype

    final val WrapperType = ObjectType.Character

    final override def isCharType: Boolean = true

    final override def computationalType: ComputationalType = ComputationalTypeInt

    final override def accept[T](v: SignatureVisitor[T]): T = v.visit(this)

    def toJava: String = "char"

    override def toBinaryJavaName: String = "C"

    override def toJVMTypeName: String = "C"

    override def toJavaClass: java.lang.Class[_] = java.lang.Character.TYPE

    override def toString() = "CharType"

    override def isWiderThan(targetType: NumericType): Boolean = false

    override def convertTo[T](
        targetType: NumericType
    )(
        implicit
        typeConversionFactory: TypeConversionFactory[T]
    ): T = {
        import typeConversionFactory._
        (targetType.id: @scala.annotation.switch) match {
            case ByteType.id                  ⇒ IntToByte
            case ShortType.id                 ⇒ IntToShort
            case CharType.id | IntegerType.id ⇒ NoConversion
            case LongType.id                  ⇒ IntToLong
            case FloatType.id                 ⇒ IntToFloat
            case DoubleType.id                ⇒ IntToDouble
        }
    }

    override def boxValue[T](
        implicit
        typeConversionFactory: TypeConversionFactory[T]
    ): T = { typeConversionFactory.PrimitiveCharToLangCharacter }

}
case object CharType extends CharType

sealed abstract class DoubleType private () extends NumericType {

    final override def isDoubleType: Boolean = true

    final override def computationalType: ComputationalType = ComputationalTypeDouble

    final override def accept[T](v: SignatureVisitor[T]): T = v.visit(this)

    final val atype = 7

    final val id = Int.MinValue + atype

    def toJava: String = "double"

    override def toBinaryJavaName: String = "D"

    override def toJVMTypeName: String = "D"

    final val WrapperType = ObjectType.Double

    override def toJavaClass: java.lang.Class[_] =
        java.lang.Double.TYPE

    override def toString() = "DoubleType"

    override def isWiderThan(targetType: NumericType): Boolean = targetType ne this

    override def convertTo[T](
        targetType: NumericType
    )(
        implicit
        typeConversionFactory: TypeConversionFactory[T]
    ): T = {
        import typeConversionFactory._
        (targetType.id: @scala.annotation.switch) match {
            case ByteType.id    ⇒ Double2Byte
            case CharType.id    ⇒ Double2Char
            case ShortType.id   ⇒ Double2Short
            case IntegerType.id ⇒ Double2Integer
            case LongType.id    ⇒ Double2Long
            case FloatType.id   ⇒ Double2Float
            case DoubleType.id  ⇒ NoConversion
        }
    }

    override def boxValue[T](
        implicit
        typeConversionFactory: TypeConversionFactory[T]
    ): T = { typeConversionFactory.PrimitiveDoubleToLangDouble }

}
case object DoubleType extends DoubleType

sealed abstract class FloatType private () extends NumericType {

    final val atype = 6

    final val id = Int.MinValue + atype

    final val WrapperType = ObjectType.Float

    final override def isFloatType: Boolean = true

    final override def computationalType: ComputationalType = ComputationalTypeFloat

    final override def accept[T](v: SignatureVisitor[T]): T = v.visit(this)

    def toJava: String = "float"

    override def toBinaryJavaName: String = "F"

    override def toJVMTypeName: String = "F"

    override def toJavaClass: java.lang.Class[_] = java.lang.Float.TYPE

    override def toString() = "FloatType"

    override def isWiderThan(targetType: NumericType): Boolean =
        (targetType ne DoubleType) && (targetType ne this)

    override def convertTo[T](
        targetType: NumericType
    )(
        implicit
        typeConversionFactory: TypeConversionFactory[T]
    ): T = {
        import typeConversionFactory._
        (targetType.id: @scala.annotation.switch) match {
            case ByteType.id    ⇒ Float2Byte
            case CharType.id    ⇒ Float2Char
            case ShortType.id   ⇒ Float2Short
            case IntegerType.id ⇒ Float2Integer
            case LongType.id    ⇒ Float2Long
            case FloatType.id   ⇒ NoConversion
            case DoubleType.id  ⇒ Float2Double
        }
    }

    override def boxValue[T](implicit typeConversionFactory: TypeConversionFactory[T]): T = {
        typeConversionFactory.PrimitiveFloatToLangFloat
    }

}
case object FloatType extends FloatType

sealed abstract class ShortType private () extends IntLikeType {

    final val atype = 9

    final val id = Int.MinValue + atype

    final val WrapperType = ObjectType.Short

    final override def isShortType: Boolean = true

    final override def computationalType: ComputationalType = ComputationalTypeInt

    final override def accept[T](v: SignatureVisitor[T]): T = v.visit(this)

    def toJava: String = "short"

    override def toBinaryJavaName: String = "S"

    override def toJVMTypeName: String = "S"

    override def toJavaClass: java.lang.Class[_] = java.lang.Short.TYPE

    override def toString() = "ShortType"

    override def isWiderThan(targetType: NumericType): Boolean = targetType eq ByteType

    override def convertTo[T](
        targetType: NumericType
    )(
        implicit
        typeConversionFactory: TypeConversionFactory[T]
    ): T = {
        import typeConversionFactory._
        (targetType.id: @scala.annotation.switch) match {
            case ByteType.id    ⇒ IntToByte
            case ShortType.id   ⇒ NoConversion
            case CharType.id    ⇒ IntToChar
            case IntegerType.id ⇒ NoConversion
            case LongType.id    ⇒ IntToLong
            case FloatType.id   ⇒ IntToFloat
            case DoubleType.id  ⇒ IntToDouble
        }
    }

    override def boxValue[T](implicit typeConversionFactory: TypeConversionFactory[T]): T = {
        typeConversionFactory.PrimitiveShortToLangShort
    }

}
case object ShortType extends ShortType

sealed abstract class IntegerType private () extends IntLikeType {

    final val atype = 10

    final val id = Int.MinValue + atype

    final val WrapperType = ObjectType.Integer

    final override def isIntegerType: Boolean = true

    final override def computationalType: ComputationalType = ComputationalTypeInt

    final override def accept[T](v: SignatureVisitor[T]): T = v.visit(this)

    def toJava: String = "int"

    override def toBinaryJavaName: String = "I"

    override def toJVMTypeName: String = "I"

    override def toJavaClass: java.lang.Class[_] = java.lang.Integer.TYPE

    override def toString() = "IntegerType"

    override def isWiderThan(targetType: NumericType): Boolean =
        (targetType.id: @scala.annotation.switch) match {
            case ShortType.id | CharType.id | ByteType.id ⇒ true
            case _                                        ⇒ false
        }

    override def convertTo[T](
        targetType: NumericType
    )(
        implicit
        typeConversionFactory: TypeConversionFactory[T]
    ): T = {
        import typeConversionFactory._
        (targetType.id: @scala.annotation.switch) match {
            case ByteType.id    ⇒ IntToByte
            case ShortType.id   ⇒ IntToShort
            case CharType.id    ⇒ IntToChar
            case IntegerType.id ⇒ NoConversion
            case LongType.id    ⇒ IntToLong
            case FloatType.id   ⇒ IntToFloat
            case DoubleType.id  ⇒ IntToDouble
        }
    }

    override def boxValue[T](implicit typeConversionFactory: TypeConversionFactory[T]): T = {
        typeConversionFactory.PrimitiveIntToLangInteger
    }

}
case object IntegerType extends IntegerType

sealed abstract class LongType private () extends NumericType {

    final val atype = 11

    final val id = Int.MinValue + atype

    final val WrapperType = ObjectType.Long

    final override def isLongType: Boolean = true

    final override def computationalType: ComputationalType = ComputationalTypeLong

    final override def accept[T](v: SignatureVisitor[T]): T = v.visit(this)

    def toJava: String = "long"

    override def toBinaryJavaName: String = "J"

    override def toJVMTypeName: String = "J"

    override def toJavaClass: java.lang.Class[_] = java.lang.Long.TYPE

    override def toString() = "LongType"

    override def isWiderThan(targetType: NumericType): Boolean =
        targetType.isInstanceOf[IntLikeType]

    override def convertTo[T](
        targetType: NumericType
    )(
        implicit
        typeConversionFactory: TypeConversionFactory[T]
    ): T = {
        import typeConversionFactory._
        (targetType.id: @scala.annotation.switch) match {
            case ByteType.id    ⇒ Long2Byte
            case CharType.id    ⇒ Long2Char
            case ShortType.id   ⇒ Long2Short
            case IntegerType.id ⇒ Long2Integer
            case LongType.id    ⇒ NoConversion
            case FloatType.id   ⇒ Long2Float
            case DoubleType.id  ⇒ Long2Double
        }
    }

    override def boxValue[T](implicit typeConversionFactory: TypeConversionFactory[T]): T = {
        typeConversionFactory.PrimitiveLongToLangLong
    }

}
case object LongType extends LongType

/**
 * The type of boolean values (true=1, false=0).
 *
 * Though the JVM internally uses an int value to store a boolean value the VM offers
 * no special further support for handling booleans. In particular the conversion of
 * some "byte|short|char|int" value to an int value is not directly supported.
 */
sealed abstract class BooleanType private () extends BaseType {

    final val atype = 4

    final val id = Int.MinValue + atype

    final val WrapperType = ObjectType.Boolean

    final override def isBooleanType: Boolean = true

    final override def computationalType: ComputationalType = ComputationalTypeInt

    final override def accept[T](v: SignatureVisitor[T]): T = v.visit(this)

    final val toJava /*: String*/ = "boolean"

    override def toBinaryJavaName: String = "Z"

    override def toJVMTypeName: String = "Z"

    override def toJavaClass: java.lang.Class[_] = java.lang.Boolean.TYPE

    override def toString() = "BooleanType"

    override def boxValue[T](implicit typeConversionFactory: TypeConversionFactory[T]): T = {
        typeConversionFactory.PrimitiveBooleanToLangBoolean
    }

}
case object BooleanType extends BooleanType

/**
 * Represents an `ObjectType`.
 *
 * @param id The unique id associated with this type.
 * @param fqn The fully qualified name of the class or interface in binary notation
 *      (e.g. "java/lang/Object").
 */
final class ObjectType private ( // DO NOT MAKE THIS A CASE CLASS!
        final val id:  Int,
        final val fqn: String
) extends ReferenceType {

    override def isObjectType: Boolean = true

    override def asObjectType: ObjectType = this

    @inline final def isPrimitiveTypeWrapper: Boolean = {
        val thisId = this.id
        thisId <= ObjectType.javaLangDoubleId && thisId >= ObjectType.javaLangBooleanId
    }

    final def isPrimitiveTypeWrapperOf(baseType: BaseType): Boolean =
        isPrimitiveTypeWrapper && (ObjectType.primitiveType(this).get eq baseType)

    def simpleName: String = ObjectType.simpleName(fqn)

    final val packageName: String = ObjectType.packageName(fqn).intern()

    override def toJava: String = fqn.replace('/', '.')

    override def toBinaryJavaName: String = "L"+toJava+";"

    override def toJVMTypeName: String = "L"+fqn+";"

    override def toJavaClass: java.lang.Class[_] = classOf[Type].getClassLoader().loadClass(toJava)

    def unboxValue[T](implicit typeConversionFactory: TypeConversionFactory[T]): T = {
        ObjectType.unboxValue(this)
    }

    override def adapt[T](
        targetType: Type
    )(
        implicit
        typeConversionFactory: TypeConversionFactory[T]
    ): T = {
        ObjectType.unboxValue(targetType)
    }

    def isSubtyeOf(that: ObjectType)(implicit classHierarchy: ClassHierarchy): Answer = {
        classHierarchy.isSubtypeOf(this, that)
    }

    // The default equals and hashCode methods are a perfect fit.

    override def toString = "ObjectType("+fqn+")"

}
/**
 * Defines factory and extractor methods for `ObjectType`s.
 *
 * @author Michael Eichberg
 */
object ObjectType {

    private[this] val nextId = new AtomicInteger(0)
    private[this] val cacheRWLock = new ReentrantReadWriteLock();
    private[this] val cache = new WeakHashMap[String, WeakReference[ObjectType]]()

    @volatile private[this] var objectTypeCreationListener: ObjectType ⇒ Unit = null

    /**
     * Sets the listener and immediately calls it (multiple times) to inform the listener
     * about all known object types. It is guaranteed that the listener will not miss any
     * object type creation. However, invocation may occur concurrently.
     */
    def setObjectTypeCreationListener(f: ObjectType ⇒ Unit): Unit = {
        cacheRWLock.readLock().lock()
        try {
            objectTypeCreationListener = f
            val objectTypesIterator = cache.values().iterator()
            while (objectTypesIterator.hasNext) {
                val objectType = objectTypesIterator.next.get()
                if (objectType ne null) f(objectType)
            }
        } finally {
            cacheRWLock.readLock().unlock()
        }
    }

    /**
     * The number of different `ObjectType`s that were created.
     */
    def objectTypesCount = nextId.get

    /**
     * Factory method to create `ObjectType`s.
     *
     * @param  fqn The fully qualified name of a class or interface type in
     *         binary notation.
     * @note   `ObjectType` objects are cached internally to reduce the overall memory
     *         requirements and to ensure that only one instance of an `ObjectType` exists
     *         per fully qualified name. Hence, comparing `ObjectTypes` using reference
     *         comparison is explicitly supported.
     */
    def apply(fqn: String): ObjectType = {
        val readLock = cacheRWLock.readLock()
        readLock.lock()
        try {
            val wrOT = cache.get(fqn)
            if (wrOT != null) {
                val OT = wrOT.get()
                if (OT != null)
                    return OT;
            }
        } finally {
            readLock.unlock()
        }

        // Remember: Lock upgrading is not possible
        val writeLock = cacheRWLock.writeLock()
        writeLock.lock()
        try {
            // WE HAVE TO CHECK AGAIN
            val wrOT = cache.get(fqn)
            if (wrOT != null) {
                val OT = wrOT.get()
                if (OT != null)
                    return OT;
            }

            val newOT = new ObjectType(nextId.getAndIncrement(), fqn)
            val wrNewOT = new WeakReference(newOT)
            cache.put(fqn, wrNewOT)
            val currentObjectTypeCreationListener = objectTypeCreationListener
            if (currentObjectTypeCreationListener ne null)
                currentObjectTypeCreationListener(newOT)
            newOT
        } finally {
            writeLock.unlock()
        }
    }

    def unapply(ot: ObjectType): Option[String] = Some(ot.fqn)

    def simpleName(fqn: String): String = {
        val index = fqn.lastIndexOf('/')
        if (index > -1)
            fqn.substring(index + 1)
        else
            fqn
    }

    /**
     * The package name of this type. The package name does not include
     * a final package separator char ("/").
     *
     * E.g.,
     * {{{
     * scala> val os = org.opalj.br.ObjectType("java/lang/String")
     * os: org.opalj.br.ObjectType = ObjectType(java/lang/String)
     *
     * scala> os.packageName
     * res1: String = java/lang
     *
     * scala> os.simpleName
     * res2: String = String
     *
     * scala> os.toJava
     * res3: String = java.lang.String
     *
     * }}}
     */
    def packageName(fqn: String): String = {
        val index = fqn.lastIndexOf('/')
        if (index == -1)
            ""
        else
            fqn.substring(0, index)
    }

    final val Object = ObjectType("java/lang/Object")
    final val ObjectId = 0
    require(Object.id == ObjectId)

    final val Boolean = ObjectType("java/lang/Boolean")

    final val Byte = ObjectType("java/lang/Byte")
    final val Character = ObjectType("java/lang/Character")
    final val Short = ObjectType("java/lang/Short")
    final val Integer = ObjectType("java/lang/Integer")
    final val Long = ObjectType("java/lang/Long")
    final val Float = ObjectType("java/lang/Float")
    final val Double = ObjectType("java/lang/Double")
    require(Double.id - Boolean.id == 7)

    final val String = ObjectType("java/lang/String")
    final val StringId = 9

    final val Class = ObjectType("java/lang/Class")
    final val ClassId = 10
    require(Class.id == 10)

    final val System = ObjectType("java/lang/System")

    final val Throwable = ObjectType("java/lang/Throwable")
    final val Error = ObjectType("java/lang/Error")
    final val Exception = ObjectType("java/lang/Exception")
    final val RuntimeException = ObjectType("java/lang/RuntimeException")

    // Types related to the invokedynamic instruction
    final val MethodHandle = ObjectType("java/lang/invoke/MethodHandle")
    final val MethodHandles$Lookup = ObjectType("java/lang/invoke/MethodHandles$Lookup")
    final val MethodType = ObjectType("java/lang/invoke/MethodType")
    final val LambdaMetafactory = ObjectType("java/lang/invoke/LambdaMetafactory")
    final val CallSite = ObjectType("java/lang/invoke/CallSite")

    // Exceptions and errors that may be thrown by the JVM (i.e., instances of these
    // exceptions may be created at runtime by the JVM)
    final val IndexOutOfBoundsException = ObjectType("java/lang/IndexOutOfBoundsException")
    final val ExceptionInInitializerError = ObjectType("java/lang/ExceptionInInitializerError")
    final val BootstrapMethodError = ObjectType("java/lang/BootstrapMethodError")
    final val OutOfMemoryError = ObjectType("java/lang/OutOfMemoryError")

    final val NullPointerException = ObjectType("java/lang/NullPointerException")
    final val ArrayIndexOutOfBoundsException = ObjectType("java/lang/ArrayIndexOutOfBoundsException")
    final val ArrayStoreException = ObjectType("java/lang/ArrayStoreException")
    final val NegativeArraySizeException = ObjectType("java/lang/NegativeArraySizeException")
    final val IllegalMonitorStateException = ObjectType("java/lang/IllegalMonitorStateException")
    final val ClassCastException = ObjectType("java/lang/ClassCastException")
    final val ArithmeticException = ObjectType("java/lang/ArithmeticException")
    final val ClassNotFoundException = ObjectType("java/lang/ClassNotFoundException")

    // the following types are relevant when checking the subtype relation between
    // two reference types where the subtype is an array type
    final val Serializable = ObjectType("java/io/Serializable")
    final val Cloneable = ObjectType("java/lang/Cloneable")

    /**
     * Least upper type bound of Java arrays. That is, every Java array
     * is always `Serializable` and `Cloneable`.
     */
    final val SerializableAndCloneable: UIDSet[ObjectType] = {
        new UIDSet2(ObjectType.Serializable, ObjectType.Cloneable)
    }

    private final val javaLangBooleanId = Boolean.id
    private final val javaLangDoubleId = Double.id

    // Given the importance of "Object Serialization" we also predefine Externalizable
    final val Externalizable = ObjectType("java/io/Externalizable")

    /**
     * Implicit mapping from a wrapper type to its primitive type.
     * @example
     * {{{
     * scala> import org.opalj.br._
     * scala> ObjectType.primitiveType(ObjectType.Integer.id)
     * res1: org.opalj.br.FieldType = IntegerType
     * }}}
     */
    private[this] lazy val primitiveType: Array[BaseType] = {
        val a = new Array[BaseType](Double.id + 1)
        a(Boolean.id) = BooleanType
        a(Byte.id) = ByteType
        a(Character.id) = CharType
        a(Short.id) = ShortType
        a(Integer.id) = IntegerType
        a(Long.id) = LongType
        a(Float.id) = FloatType
        a(Double.id) = DoubleType
        a
    }

    def unboxValue[T](
        wrapperType: Type
    )(
        implicit
        typeConversionFactory: TypeConversionFactory[T]
    ): T = {
        typeConversionFactory.unboxValue(wrapperType)
    }

    /**
     * Given a wrapper type (e.g., `java.lang.Integer`) the underlying primitive type
     * is returned.
     *
     * @example
     * {{{
     * scala> import org.opalj.br._
     * scala> ObjectType.primitiveType(ObjectType.Integer)
     * res0: Option[org.opalj.br.BaseType] = Some(IntegerType)
     * }}}
     */
    def primitiveType(wrapperType: ObjectType): Option[BaseType] = {
        val wrapperId = wrapperType.id
        if (wrapperId < 0 || wrapperId > Double.id) {
            None
        } else {
            Some(primitiveType(wrapperId))
        }
    }

    def primitiveTypeWrapperMatcher[Args, T](
        booleanMatch: (Args) ⇒ T,
        byteMatch:    (Args) ⇒ T,
        charMatch:    (Args) ⇒ T,
        shortMatch:   (Args) ⇒ T,
        integerMatch: (Args) ⇒ T,
        longMatch:    (Args) ⇒ T,
        floatMatch:   (Args) ⇒ T,
        doubleMatch:  (Args) ⇒ T,
        orElse:       (Args) ⇒ T
    ): (ObjectType, Args) ⇒ T = {
        val fs = new Array[(Args) ⇒ T](8)
        fs(0) = booleanMatch
        fs(1) = byteMatch
        fs(2) = charMatch
        fs(3) = shortMatch
        fs(4) = integerMatch
        fs(5) = longMatch
        fs(6) = floatMatch
        fs(7) = doubleMatch

        (objectType: ObjectType, args: Args) ⇒ {
            val oid = objectType.id
            if (oid > javaLangDoubleId || oid < javaLangBooleanId) {
                orElse(args)
            } else {
                val index = oid - javaLangBooleanId
                fs(index)(args)
            }
        }
    }

    @inline final def isPrimitiveTypeWrapper(objectType: ObjectType): Boolean = {
        val oid = objectType.id
        oid <= javaLangDoubleId && oid >= javaLangBooleanId
    }
}

/**
 * Represents an array type.
 *
 * ==Comparing `ArrayType`s==
 * To facilitate comparisons of (array) types, each array type is represented
 * at any given time, by exactly one instance of `ArrayType`.
 *
 * ==General Information==
 * ''From the JVM specification''
 *
 * An array type consists of a '''component type''' with a single dimension (whose length is
 * not given by the type). The component type of an array type may itself be an array
 * type. If, starting from any array type, one considers its component type, and then
 * (if that is also an array type) the component type of that type, and so on, eventually
 * one must reach a component type that is not an array type; this is called the '''element
 * type of the array type'''. The element type of an array type is necessarily either a
 * primitive type, or a class type, or an interface type.
 *
 * @author Michael Eichberg
 */
final class ArrayType private ( // DO NOT MAKE THIS A CASE CLASS!
        val id:            Int,
        val componentType: FieldType
) extends ReferenceType {

    final override def isArrayType = true

    final override def asArrayType = this

    /**
     * Returns this array type's element type. E.g., the element type of an
     * array of arrays of arrays of `int` is `int`.
     */
    def elementType: FieldType = {
        componentType match {
            case at: ArrayType ⇒ at.elementType
            case _             ⇒ componentType
        }
    }

    /**
     * The number of dimensions of this array. E.g. "Object[]" has one dimension and
     * "Object[][]" has two dimensions.
     */
    def dimensions: Int = {
        1 + (componentType match { case at: ArrayType ⇒ at.dimensions; case _ ⇒ 0 })
    }

    /**
     * Returns the component type of this array type after dropping the given number
     * of dimensions. E.g., if dimensions is `0`
     * `this` is returned; if it is `1` then this arraytype's component type is returned.
     * If the value is larger than `1` then the `componentType` has to be an array type
     * and `drop(dimensions-1)` will be called on that type.
     *
     * @param dimensions The number of dimensions to drop. This values has be equal or
     *      smaller than the number of dimensions of this array.
     */
    def drop(dimensions: Int): FieldType = {
        dimensions match {
            case 0 ⇒ this
            case 1 ⇒ this.componentType
            case _ ⇒ this.componentType.asArrayType.drop(dimensions - 1)
        }
    }

    override def toJava: String = componentType.toJava+"[]"

    override def toBinaryJavaName: String = "["+componentType.toBinaryJavaName

    override def toJVMTypeName: String = "["+componentType.toJVMTypeName

    override def toJavaClass: java.lang.Class[_] = java.lang.Class.forName(toBinaryJavaName)

    override def adapt[T](
        targetType: Type
    )(
        implicit
        typeConversionFactory: TypeConversionFactory[T]
    ): T = {
        throw new UnsupportedOperationException("adaptation of array values is not supported")
    }

    // The default equals and hashCode methods are a perfect fit.

    override def toString = "ArrayType("+componentType.toString+")"
}

/**
 * Defines factory and extractor methods for `ArrayType`s.
 *
 * @author Michael Eichberg
 */
object ArrayType {

    private[this] val cache = new WeakHashMap[FieldType, WeakReference[ArrayType]]()

    private[this] val nextId = new AtomicInteger(-1)

    /**
     * Factory method to create objects of type `ArrayType`.
     *
     * ==Note==
     * `ArrayType` objects are cached internally to reduce the overall memory requirements
     * and to facilitate reference based comparisons. I.e., to `ArrayType`s are equal
     * iff it is the same object.
     */
    def apply(componentType: FieldType): ArrayType = {
        cache.synchronized {
            val wrAT = cache.get(componentType)
            if (wrAT != null) {
                val AT = wrAT.get()
                if (AT != null)
                    return AT;
            }
            val newAT = new ArrayType(nextId.getAndDecrement(), componentType)
            val wrNewAT = new WeakReference(newAT)
            cache.put(componentType, wrNewAT)
            newAT
        }
    }

    /**
     * Factory method to create an Array of the given component type with the given
     * dimension.
     */
    @tailrec def apply(dimension: Int, componentType: FieldType): ArrayType = {
        assert(dimension >= 1, s"dimension=$dimension, componentType=$componentType")

        val at = apply(componentType)
        if (dimension > 1)
            apply(dimension - 1, at)
        else
            at
    }

    def unapply(at: ArrayType): Option[FieldType] = Some(at.componentType)

    final val ArrayOfObjects = ArrayType(ObjectType.Object)
}

/**
 * Facilitates matching against an array's element type.
 *
 * @author Michael Eichberg
 */
object ArrayElementType {
    def unapply(at: ArrayType): Option[FieldType] = Some(at.elementType)
}

/**
 * Defines an extractor to match a type against any `ObjectType`
 * except `java.lang.Object`.
 *
 * @author Michael Eichberg
 */
object NotJavaLangObject {

    def unapply(objectType: ObjectType): Boolean = objectType ne ObjectType.Object
}

/**
 * Defines an extractor to match against any `Type` except `void`. Can be useful, e.g.,
 * when matching `MethodDescriptor`s to select all methods that return something.
 *
 * @author Michael Eichberg
 */
object NotVoid {

    def unapply(someType: Type): Boolean = someType ne VoidType

}
