/* BSD 2-Clause License:
 * Copyright (c) 2009 - 2017
 * Software Technology Group
 * Department of Computer Science
 * Technische Universität Darmstadt
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package org.opalj
package da

import scala.xml.Node

import org.opalj.bi.AccessFlags
import org.opalj.bi.AccessFlagsContexts.MODULE

/**
 * Representation of the ''Module'' attribute (Java 9).
 *
 * @author Michael Eichberg
 */
case class Module_attribute(
        attribute_name_index: Constant_Pool_Index,
        requires:             IndexedSeq[RequiresEntry],
        exports:              IndexedSeq[ExportsEntry],
        uses:                 IndexedSeq[UsesEntry],
        provides:             IndexedSeq[ProvidesEntry]
) extends Attribute {

    def attribute_length: Int = {
        2 + requires.length * 4 + // <= requires
            2 + exports.length * 8 + // <= exports
            2 + uses.length * 2 + // <= uses
            2 + provides.length * 4 // <= provides
    }

    override def toXHTML(implicit cp: Constant_Pool): Node = {
        <details>
            <summary>Module</summary>
            <div>{ requires.map(_.toString(cp)).sorted.map(r ⇒ <span>{ r }</span><br/>) }</div>
            <div>{ exports.map(_.toString(cp)).sorted.map(r ⇒ <span>{ r }</span><br/>) }</div>
            <div>{ uses.map(_.toString(cp)).sorted.map(r ⇒ <span>{ r }</span><br/>) }</div>
            <div>{ provides.map(_.toString(cp)).sorted.map(r ⇒ <span>{ r }</span><br/>) }</div>
        </details>
    }

}

case class RequiresEntry(
        requires_index: Constant_Pool_Index, // CONSTANT_UTF8
        requires_flags: Int
) {

    def toString(implicit cp: Constant_Pool) = {
        val flags = AccessFlags.toString(requires_flags, MODULE)
        s"requires ${flags} ${cp(requires_index).toString(cp)};"
    }
}

case class ExportsEntry(
        exports_index: Constant_Pool_Index, // CONSTANT_UTF8
        // TODO Documented in JSR by not yet(?) generated by the JDK 9 javac (Aug. 2016): exports_flags: Int,
        exports_to: IndexedSeq[ExportsToEntry]
) {

    def toString(implicit cp: Constant_Pool) = {
        val flags =
            // TODO Documented in JSR by not yet(?) generated by the JDK 9 javac (Aug. 2016): AccessFlags.toString(exports_flags, MODULE)
            // TODO Use above or delete next row when JDK 9 is finalized
            ""
        val to = {
            if (exports_to.isEmpty)
                ";"
            else
                exports_to.map(_.toString).toList.sorted.mkString(" to ", ", ", ";")
        }

        s"exports $flags ${cp(exports_index).toString(cp)}$to"
    }
}

case class ExportsToEntry(
        exports_to_index: Constant_Pool_Index // CONSTANT_UTF8
) {

    def toString(implicit cp: Constant_Pool): String = {
        cp(exports_to_index).toString(cp)
    }
}

case class UsesEntry(
        uses_index: Constant_Pool_Index // CONSTANT_Class
) {

    def toString(implicit cp: Constant_Pool): String = {
        s"uses ${cp(uses_index).toString(cp)};"
    }

}

case class ProvidesEntry(
        provides_index: Constant_Pool_Index, // CONSTANT_Class
        with_index:     Constant_Pool_Index
) {

    def toString(implicit cp: Constant_Pool): String = {
        s"provides ${cp(provides_index).toString(cp)} with ${cp(with_index).toString(cp)};"
    }
}
