/* BSD 2-Clause License:
 * Copyright (c) 2009 - 2017
 * Software Technology Group
 * Department of Computer Science
 * Technische Universität Darmstadt
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package org.opalj
package da

import scala.xml.Node
import scala.xml.NodeSeq

/**
 * @author Michael Eichberg
 * @author Wael Alkhatib
 * @author Isbel Isbel
 * @author Noorulla Sharief
 * @author Andre Pacak
 */
case class Code_attribute(
        attribute_name_index: Constant_Pool_Index,
        max_stack:            Int,
        max_locals:           Int,
        code:                 Code,
        exceptionTable:       ExceptionTable      = IndexedSeq.empty,
        attributes:           Attributes          = IndexedSeq.empty
) extends Attribute {

    override def attribute_length: Int = {
        2 + 2 +
            4 /*code_length*/ + code.instructions.length +
            2 /*exception_table_length*/ + 8 * exceptionTable.length +
            2 /*attributes_count*/ + attributes.foldLeft(0)(_ + _.size)
    }

    /**
     * @see `toXHTML(Int)(implicit Constant_Pool)`
     */
    @throws[UnsupportedOperationException]("always")
    override def toXHTML(implicit cp: Constant_Pool): Node = {
        val message = "use toXHTML(methodIndex: Int)(implicit cp: Constant_Pool)"
        throw new UnsupportedOperationException(message)
    }

    def toXHTML(methodIndex: Int)(implicit cp: Constant_Pool): Node = {
        val codeSize = code.instructions.size
        val methodBodyHeader =
            s"Method Body (Size: $codeSize bytes, Max Stack: $max_stack, Max Locals: $max_locals)"
        <details class="method_body">
            <summary>{ methodBodyHeader }</summary>
            {
                code.toXHTML(
                    methodIndex,
                    exceptionTable,
                    attributes.collectFirst { case LineNumberTable_attribute(_, lnt) ⇒ lnt }
                )
            }
            { exception_handlersAsXHTML }
            { attributesAsXHTML }
        </details>

    }

    def attributesAsXHTML(implicit cp: Constant_Pool): Seq[Node] = attributes.map(_.toXHTML)

    /** Can only be called if the exception table is non-emtpy! */
    def exception_handlersAsXHTML(implicit cp: Constant_Pool): NodeSeq = {
        if (exceptionTable.length > 0)
            <details class="exception_table">
                <summary>Exception Table:</summary>
                <ol class="exception_table">
                    { exceptionTable.map(_.toXHTML(cp, code)) }
                </ol>
            </details>
        else
            NodeSeq.Empty
    }
}
