package de.oberdorf_itc.pushover;

import java.util.HashMap;
import java.util.Set;
import net.pushover.client.*;
import net.pushover.client.PushoverMessage.Builder;

/**
 * Class Client - A wrapper class for the net.pushover.client with the ability to send the pushover message in a background thread
 * Copyright by Michael Oberdorf IT-Consulting 2017<br>
 * Date: 2017-10-19
 * @author Michael Oberdorf
 * @version 0.3.0
 * @see "https://www.oberdorf-itc.de/"
 * @see "https://maven.apache.org/archetype/maven-archetype-plugin/specification/archetype-catalog.html"
 * @see "https://github.com/sps/pushover4j"
 */
public class Client implements Runnable
  {
  private PushoverClient POC = new PushoverRestClient();
  protected String apiToken = null;
  protected String userID = null;
  private MessagePriority messagePriority = MessagePriority.NORMAL;
  private String title = null;
  private String url = null;
  private String urlTitle = null;
  private String device = null;
  private String body = null;
  private String sound = null;
  private String requestID = null;
  private int status;
  private String status_text = null;
  
  /**
   * Client constructor
   */
  public Client() { }

  /**
   * setApiToken - sets the Pushover API Token ID
   * @param apiToken (String, Pushover API Token ID)
   */
  public void setApiToken(String apiToken)
    {
	this.apiToken = apiToken;
    }
  
  /**
   * setUserId - sets the Pushover User ID
   * @param userID (String, Pushover User ID)
   */
  public void setUserId(String userID)
    {
	this.userID = userID;
    }
  
  /**
   * setMessagePriority - sets the Pushover Message Priority
   * @param priority (String, Pushover Message Priority, one of "QUIET", "NORMAL", "HIGH", default "NORMAL")
   */
  public void setMessagePriority(String priority)
    {
	priority = priority.toUpperCase().trim();
	switch(priority)
	  {
	  case "QUIET":
		this.messagePriority = MessagePriority.QUIET;
		break;
	  case "HIGH":
		this.messagePriority = MessagePriority.HIGH;
		break;
	  default:
	    this.messagePriority = MessagePriority.NORMAL;
		break;
	  }
    }  
  
  /**
   * setTitle - sets the Pushover Message title
   * @param title (String, Pushover Message title)
   */
  public void setTitle(String title)
    {
	this.title = title;
    }
  
  /**
   * setURL - sets a Pushover URL to append to message
   * @param url (String, Pushover URL)
   * @param urlTitle (String, Title for the Pushover URL)
   */
  public void setURL(String url, String urlTitle)
    {
	this.url = url;
	this.urlTitle = urlTitle;
    }

  /**
   * setDevice - defines to send message to specific device
   * @param deviceName (String, device name to send)
   */
  public void setDevice(String deviceName)
    {
	this.device = deviceName;
    }
 
  /**
   * setBody - set the Pushover message to send
   * @param body (String, Pushover message text)
   */
  public void setBody(String body)
    {
	this.body = body;
    }
  
  /**
   * setSound - set the Pushover message sound
   * @param sound (String, Pushover message sound)
   */
  public void setSound(String sound)
    {
	this.sound = sound;
    }
  
  
  /**
   * pushMessage - Push the message
   * @return int (Status, -1 on missing data)
   */
  public int pushMessage()
    {
	if (this.apiToken == null)
	  {
	  this.status = -1;
	  this.status_text = "apiToken missing";
	  return this.status;
	  }
	if (this.userID == null)
	  {
	  this.status = -1;
	  this.status_text = "userID missing";
	  return this.status;
	  }
	
	// build the basic message object
	Builder data =  PushoverMessage.builderWithApiToken(this.apiToken).setUserId(this.userID).setPriority(this.messagePriority);

	// add message title if available
	if (this.title != null) { data.setTitle(this.title); }
	
    // add message text
	if(this.body != null) { data.setMessage(this.body); }
	
	// set URL
	if (this.url != null) { data.setUrl(this.url); }

	// set URL title
	if (this.urlTitle != null) { data.setTitleForURL(this.urlTitle); }
	
	// set sound
	if (this.sound != null) { data.setSound(this.sound); }
	
	// set Device
	if (this.device != null) { data.setDevice(this.device); }
	
	// push the message
	Status result = null;
	try
	  {
	  result = this.POC.pushMessage(data.build());
	  }
	catch (PushoverException e)
	  {
	  this.status = -1;
	  this.status_text = e.toString();
	  return this.status;
	  }
	
	// read te status
	this.status = result.getStatus();
	this.requestID = result.getRequestId();
	
	return this.status;
    }
  
  /**
   * send pushover message in background thread
   */
  public void run() { pushMessage(); }
  
  
  /**
   * getLastStatus - returns the last status
   * @return int (Status)
   */
  public int getLastStatus()
    {
	return this.status;  
    }
 
  /**
   * getLastStatusText - returns the text description of the last status
   * @return String (Status text)
   */
  public String getLastStatusText()
    {
	return this.status_text;  
    }
  
  /**
   * getLastRequestID - returns the request ID from last push
   * @return String (Pushover Request ID)
   */
  public String getLastRequestID()
    {
	return this.requestID;  
    }
  
  /**
   * getSoundList - returns a list of available Pushover sounds
   * @return HashMap&lt;String, String&gt; (HashMap of available sound IDs with their correponding names)
   */
  public HashMap<String, String> getSoundList()
    {
	HashMap<String, String> soundList = new HashMap<String,String>();
	
	Set<PushOverSound> sounds = null;
	try { sounds = this.POC.getSounds(); }
	catch (PushoverException e)
	  {
	  this.status = -1;
	  this.status_text = e.toString();
	  return soundList;
	  }
    this.status = 0;
	this.status_text = "Successfully fetching sounds";

	
	// loop over sounds and return the sound names
	for (PushOverSound sound : sounds)
	  {
	  soundList.put(sound.getId(), sound.getName());
	  }
	
	return soundList;
    }
  }
