package de.oberdorf_itc.mqtt;

import java.io.IOException;
import org.eclipse.paho.client.mqttv3.MqttClient;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.eclipse.paho.client.mqttv3.MqttMessage;
import org.eclipse.paho.client.mqttv3.persist.MemoryPersistence;

/**
 * Class MQTTPublish - to encapsulate MQTT Paho MQTT client class as runable thread
 * Copyright by Michael Oberdorf IT-Consulting 2018,<br>
 * Date: 2018-09-30
 * @author Michael Oberdorf
 * @version 0.100
 * @see "https://www.oberdorf-itc.de/"
 * @see "https://www.eclipse.org/paho/files/javadoc/org/eclipse/paho/client/mqttv3/MqttClient.html"
 */
public class MQTTPublish implements Runnable
  {
  private MqttClient mqttClient=null;
  private String mqttUser=null;
  protected char[] mqttPass=null;
  private int qos = 0;
  private boolean retained = false;
  private String topic=null;
  private String body=null;
  
  /**
   * MQTTPublish constructor
   * @param server (String: the MQTT server connection in form [tcp|ssl]://&lt;url&gt;:port)
   * @throws MqttException if the MqttClient can't be initialized
   */
  public MQTTPublish(String server) throws MqttException
    {
	MemoryPersistence persistence = new MemoryPersistence();
	String clientId = MqttClient.generateClientId();
	this.mqttClient = new MqttClient(server, clientId, persistence);
    }

  /**
   * MQTTPublish constructor
   * @param server (String: the MQTT server connection in form [tcp|ssl]://&lt;url&gt;:port)
   * @param mqttUser (String: the username for authentication)
   * @param mqttPass (String: the password for authentication)
   * @throws MqttException if the MqttClient can't be initialized
   */
  public MQTTPublish(String server, String mqttUser, String mqttPass) throws MqttException
    {
	MemoryPersistence persistence = new MemoryPersistence();
	String clientId = MqttClient.generateClientId();
	this.mqttClient = new MqttClient(server, clientId, persistence);
	this.mqttUser = mqttUser;
	this.mqttPass = mqttPass.toCharArray();
    }
  
  /**
   * MQTTPublish constructor
   * @param server (String: the MQTT server connection in form [tcp|ssl]://&lt;url&gt;:port)
   * @param clientId (String: the ClientId to use)
   * @param mqttUser (String: the username for authentication)
   * @param mqttPass (String: the password for authentication)
   * @throws MqttException if the MqttClient can't be initialized
   */
  public MQTTPublish(String server, String clientId, String mqttUser, String mqttPass) throws MqttException
    {
	MemoryPersistence persistence = new MemoryPersistence();
	this.mqttClient = new MqttClient(server, clientId, persistence);
	this.mqttUser = mqttUser;
	this.mqttPass = mqttPass.toCharArray();
    }	

  /**
   * setPublishMessage - method to set the message to publish 
   * @param topic (String: the MQTT topic where the message should be published)
   * @param body (String: the MQTT message that should be published)
   */
  public void setPublishMessage(String topic, String body)
    {
	this.topic = topic;
	this.body = body;
	this.qos = 0;
	this.retained = false;
    }

  /**
   * setPublishMessage - method to set the message to publish 
   * @param topic (String: the MQTT topic where the message should be published)
   * @param body (String: the MQTT message that should be published)
   * @param qos (int: the QoS for the publish, valid values are: 0, 1, 2)
   */
  public void setPublishMessage(String topic, String body, int qos)
    {
	this.topic = topic;
	this.body = body;
	this.qos = qos;
	this.retained = false;
    }

  /**
   * setPublishMessage - method to set the message to publish 
   * @param topic (String: the MQTT topic where the message should be published)
   * @param body (String: the MQTT message that should be published)
   * @param retained (Boolean: should the message be retained on the MQTT Broker)
   */
  public void setPublishMessage(String topic, String body, boolean retained)
    {
	this.topic = topic;
	this.body = body;
	this.qos = 0;
	this.retained = retained;
    }
  
  /**
   * setPublishMessage - method to set the message to publish 
   * @param topic (String: the MQTT topic where the message should be published)
   * @param body (String: the MQTT message that should be published)
   * @param qos (int: the QoS for the publish, valid values are: 0, 1, 2)
   * @param retained (Boolean: should the message be retained on the MQTT Broker)
   */
  public void setPublishMessage(String topic, String body, int qos, boolean retained)
    {
	this.topic = topic;
	this.body = body;
	this.qos = qos;
	this.retained = retained;
    }
  

  /**
   * run - overwrites the run method to start the tread in background to send the mail
   */
  @Override
  public void run()
    {
    // check if all mandatory params has been set
	try
	  {
	  if (this.mqttClient == null) { throw new IOException(); }
	  if (this.topic == null) { throw new IOException(); }
	  if (this.body == null) { throw new IOException(); }
	  }
	catch(IOException e)
	  {
	  e.printStackTrace();
	  }
    try
      {
      MqttConnectOptions connOpts = new MqttConnectOptions();
      if (this.mqttUser != null && this.mqttPass != null)
        {
	    connOpts.setUserName(mqttUser);
	    connOpts.setPassword(mqttPass);
        }
	  connOpts.setCleanSession(true);
	  this.mqttClient.connect(connOpts);
	  MqttMessage message = new MqttMessage(this.body.getBytes());
      message.setQos(this.qos);
	  message.setRetained(this.retained);
	  this.mqttClient.publish(topic, message);
	  this.mqttClient.disconnect();
      }
    catch(MqttException e)
      {
      /*
      System.out.println("reason "+e.getReasonCode());
      System.out.println("msg "+e.getMessage());
      System.out.println("loc "+e.getLocalizedMessage());
      System.out.println("cause "+e.getCause());
      System.out.println("excep "+e);
      */
      e.printStackTrace();
	  }
	}
}