package de.mvbonline.tools.restapidoc;

import java.io.IOException;

import javax.xml.bind.JAXB;

import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.AnnotationConfigApplicationContext;
import org.springframework.core.io.Resource;

import de.mvbonline.tools.restapidoc.doclet.model.ApiDescription;
import de.mvbonline.tools.restapidoc.doclet.model.ClassDescription;
import de.mvbonline.tools.restapidoc.doclet.model.ElementDescription;
import de.mvbonline.tools.restapidoc.doclet.model.MethodDescription;

/**
 * loads all apidoc.xml (generated by  {@link JavadocToXMLDoclet} found in classpath and offers convenience methods for access.
 * 
 */
public class ApiDescriptionFinder {

    private ApiDescription apiDescriptions;
    
    
    public ApiDescriptionFinder() throws IOException{
        super();
        collectApiDescriptions();
    }


    /**
     * Looks for apidoc.xml Javadoc description files (generated by RestApiDoclet) in the classpath and loads them.
     * 
     * @throws IOException
     */
    private void collectApiDescriptions() throws IOException {
        ApplicationContext applicationContext = new AnnotationConfigApplicationContext();
        Resource[] apidocsRes = applicationContext.getResources("classpath*:/META-INF/apidocs/apidoc.xml");
        apiDescriptions = new ApiDescription();
        for (Resource apidocRes : apidocsRes) {
            ApiDescription apidoc = JAXB.unmarshal(apidocRes.getInputStream(), ApiDescription.class);
            apiDescriptions.addClasses(apidoc.getClasses());
        }
    }
    

    /**
     * looks up a class description
     * 
     * @param className
     *            full qualified name of the class
     * @return the ClassDescription or null if none were found
     */
    public ClassDescription getClassDescription(String className) {
        if (apiDescriptions != null && apiDescriptions.getClasses() != null) {
            for (ClassDescription classDoc : apiDescriptions.getClasses()) {
                if (classDoc.getFullClassName().equals(className)) {
                    return classDoc;
                }
            }
        }
        return null;
    }

    /**
     * looks up an element description
     * @param methodDesc
     *            the mthod description
     * @param name
     *            of the parameter
     * @return the description or an empty string if none were found
     */
    public String getElementDescription(MethodDescription methodDesc, String name) {
        if (methodDesc != null && methodDesc.getParameters() != null) {
            for (ElementDescription elementDoc : methodDesc.getParameters()) {
                if (elementDoc.getName().equals(name)) {
                    return elementDoc.getDescription();
                }
            }
        }
        return "";
    }

    /**
     * looks up a description of a field/property
     * @param classDesc the owning class
     * @param name of the property/field
     * @return either the description of the field or an empty string if it doesn't exist
     */
    public String getFieldDescription(ClassDescription classDesc, String name) {
        if (classDesc != null && classDesc.getProperties() != null) {
            for (ElementDescription elementDoc : classDesc.getProperties()) {
                if (elementDoc.getName().equals(name)) {
                    return elementDoc.getDescription();
                }
            }
        }
        return "";
    }
    
    /**
     * looks up a method description
     * 
     * @param classDesc
     *            the class description containing the method.
     * @param name
     *            of the method
     * @return methodDescription or null if none were found
     */
    public MethodDescription getMethodDescription(ClassDescription classDesc, String name) {
        if (classDesc != null) {
            for (MethodDescription methodDoc : classDesc.getMethods()) {
                if (methodDoc.getName().equals(name)) {
                    return methodDoc;
                }
            }
        }
        return null;
    }
}
