package de.monochromata.cucumber.stepdefs;

import static org.assertj.core.api.Assertions.assertThat;

import java.io.PrintWriter;
import java.io.StringWriter;

import io.cucumber.docstring.DocString;
import io.cucumber.java.en.Then;

public class ExceptionStepdefs {

    private final ExceptionState exceptionState;

    public ExceptionStepdefs(final ExceptionState exceptionState) {
        this.exceptionState = exceptionState;
    }
    
    @Then("there is no exception")
    @Then("no exception is raised/thrown")
    public void assertNoExceptionIsRaised() {
        assertThat(getStackTrace()).isEmpty();
    }

    @Then("a(n) {word} is raised/thrown")
    public void assertException(final String simpleTypeName) {
        assertThat(exceptionState.exception).isNotNull();
        assertThat(exceptionState.exception.getClass().getSimpleName())
                .as("unexpected exception type in " + getStackTrace()).isEqualTo(simpleTypeName);
    }
    
    @Then("the exception is a runtime exception")
    public void assertIsRuntimeException() {
        assertThat(exceptionState.exception).isInstanceOf(RuntimeException.class);
    }

    @Then("a(n) {word} is raised/thrown with message:")
    public void assertExceptionWithMessageDocString(final String simpleTypeName, final DocString message) {
        assertExceptionWithMessage(simpleTypeName, message.getContent());
    }
    
    @Then("a(n) {word} is raised/thrown with message {string}")
    @Then("a(n) {word} with message {string} is raised/thrown")
    public void assertExceptionWithMessage(final String simpleTypeName, final String message) {
        assertException(simpleTypeName);
        assertThat(exceptionState.exception.getMessage()).isEqualTo(message);
    }
    
    @Then("a(n) {word} is raised/thrown with message matching {string}")
    @Then("a(n) {word} with message matching {string} is raised/thrown")
    public void assertExceptionWithMessageRegex(final String simpleTypeName, final String messageRegex) {
        assertException(simpleTypeName);
        assertThat(exceptionState.exception.getMessage()).matches(messageRegex);
    }
    
    @Then("the {word} has been caused by a(n) {word}")
    public void assertExceptionCause(final String simpleTypeName, final String simpleCauseTypeName) {
        assertException(simpleTypeName);
        final var cause = exceptionState.exception.getCause();
        assertThat(cause).isNotNull();
        assertThat(cause.getClass().getSimpleName()).as("unexpected exception cause type in " + getStackTrace(cause))
                .isEqualTo(simpleCauseTypeName);
    }
    
    @Then("a(n) {word} is raised/thrown that has been caused by a(n) {word} with message {string}")
    public void assertExceptionCauseWithMessage(final String simpleTypeName, final String simpleCauseTypeName,
            final String causeMessage) {
        assertExceptionCause(simpleTypeName, simpleCauseTypeName);
        final var cause = exceptionState.exception.getCause();
        assertThat(cause.getMessage()).as("unexpected exception cause message in " + getStackTrace(cause))
                .isEqualTo(causeMessage);
    }
    
    @Then("a(n) {word} is raised/thrown that has been caused by a(n) {word} with message matching {string}")
    public void assertExceptionCauseWithMessageRegex(final String simpleTypeName, final String simpleCauseTypeName,
            final String causeMessageRegex) {
        assertExceptionCause(simpleTypeName, simpleCauseTypeName);
        final var cause = exceptionState.exception.getCause();
        assertThat(cause.getMessage()).as("unexpected exception cause message in " + getStackTrace(cause))
            .matches(causeMessageRegex);
    }

    protected String getStackTrace() {
        final var exception = exceptionState.exception;
        return getStackTrace(exception);
    }

    protected String getStackTrace(final Throwable exception) {
        final var writer = new StringWriter();
        if (exception != null) {
            exception.printStackTrace(new PrintWriter(writer));
        }
        return writer.toString();
    }

}
