package de.monochromata.cucumber.stepdefs;

import static java.nio.file.Files.createDirectories;
import static java.nio.file.Files.createTempDirectory;
import static java.nio.file.Files.writeString;
import static java.util.stream.Collectors.joining;
import static org.eclipse.jdt.core.compiler.batch.BatchCompiler.compile;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Stream;

import io.cucumber.docstring.DocString;
import io.cucumber.java.en.Given;
import io.cucumber.java.en.When;

@SuppressWarnings({ "rawtypes", "unchecked" })
public class JavaCompilerStepdefs {

    private final JavaCompilerState<Object> state;
    private final ExceptionState exceptionState;

    public JavaCompilerStepdefs(
            final JavaCompilerState state, 
            final ExceptionState exceptionState) {
        this.state = state;
        this.exceptionState = exceptionState;
    }
    
    @Given("a class {string} from source:")
    public void aClassFromSource(final String className, final DocString javaSource) {
        try {
            var outputDir = compileClass(className, javaSource.getContent());
            state.clazz = defineClass(className, outputDir, true);
        } catch (final Exception e) {
            if(state.catchExceptionsForAssertions) {
                exceptionState.exception = e;
            } else {
                throw new RuntimeException("Failed to compile/load class " + className + ", see standard error", e);
            }
        }
    }
    
    @Given("classes {string} from source:")
    public void classesFromSource(final String commaSeparatedClassNames, final DocString sources) {
        try {
            final var classNames = commaSeparatedClassNames.split(",");
            final var documents = sources.getContent().split("---");
            var outputDir = compileClasses(classNames, documents);
            state.classes = defineClasses(outputDir, true, classNames);
        } catch (final Exception e) {
            if(state.catchExceptionsForAssertions) {
                exceptionState.exception = e;
            } else {
                throw new RuntimeException("Failed to compile/load classes " + commaSeparatedClassNames + ", see standard error", e);
            }
        }
    }
    
    @Given("a class {string} from source defined by a class loader that does not delegate to its parent:")
    public void aClassFromSourceOSGI(final String className, final DocString javaSource) {
        try {
            var outputDir = compileClass(className, javaSource.getContent());
            state.clazz = defineClass(className, outputDir, false);
        } catch (final Exception e) {
            if(state.catchExceptionsForAssertions) {
                exceptionState.exception = e;
            } else {
                throw new RuntimeException("Failed to compile/load class "+className+", see standard error", e);
            }
        }
    }
    
    @When("an instance of the class is created")
    public void anInstanceOfTheClassIsCreated() {
        try {
            state.instance = state.clazz.getDeclaredConstructor().newInstance();
        } catch (final Exception e) {
            if(state.catchExceptionsForAssertions) {
                exceptionState.exception = e;
            } else {
                throw new RuntimeException("Failed to instantiate class via no-args constructor", e);
            }
        }
    }
    
    @When("an instance of {string} is created")
    public void anInstanceIsCreated(final String typeName) {
        try {
            final var clazz = state.classes.get(typeName);
            final var instance = clazz.getDeclaredConstructor().newInstance();
            state.instances.put(typeName, instance);
        } catch (final Exception e) {
            if(state.catchExceptionsForAssertions) {
                exceptionState.exception = e;
            } else {
                throw new RuntimeException("Failed to instantiate class via no-args constructor", e);
            }
        }
    }

    protected Path compileClasses(final String[] classNames, final String[] sources) throws IOException {
        var sourceFiles = saveSources(classNames, sources)
                .map(Path::toAbsolutePath)
                .map(Path::toString)
                .collect(joining(" "));
        var outputDir = createTempDirectory("outputDir");
        if(!compile("-11 " + sourceFiles + " -d "+outputDir, new PrintWriter(System.out), new PrintWriter(System.err), null)) {
            throw new RuntimeException("Compilation failed");
        }
        return outputDir;
    }
    
    protected Path compileClass(final String className, final String javaSource) throws IOException {
        var sourceFile = saveSource(className, javaSource);
        var outputDir = createTempDirectory("outputDir");
        if(!compile("-11 " + sourceFile + " -d "+outputDir, new PrintWriter(System.out), new PrintWriter(System.err), null)) {
            throw new RuntimeException("Compilation failed");
        }
        return outputDir;
    }
    
    protected Stream<Path> saveSources(final String[] classNames, final String[] sources) throws IOException {
        if(classNames.length != sources.length) {
            throw new IllegalArgumentException("Mismatch: you defined " + classNames.length 
                    + " class name(s) but the source file is divided into " + sources.length + " source fragment(s)"
                    + " - they should match");
        }
        final Stream.Builder<Path> sourceFiles = Stream.builder();
        var inputDir = createTempDirectory("inputDir");
        for(int i=0;i<classNames.length;i++) {
            sourceFiles.add(saveSource(classNames[i], sources[i], inputDir));
        }
        return sourceFiles.build();
    }

	protected Path saveSource(final String className, final String javaSource) throws IOException {
		var inputDir = createTempDirectory("inputDir");
		return saveSource(className, javaSource, inputDir);
	}

    public Path saveSource(final String className, final String javaSource, Path inputDir) throws IOException {
        var filename = convertPackageToDirectories(className, ".java");
		var file = inputDir.resolve(filename);
		createDirectories(file.getParent());
		return writeString(file, javaSource);
    }
	
	protected Map<String,Class<?>> defineClasses(final Path outputDir,
	        final boolean delegateToParentClassLoader,
	        final String... classNames) throws IOException {
	    final var classes = new HashMap<String,Class<?>>();
	    for(final String className: classNames) {
	        final var clazz = defineClass(className, outputDir, delegateToParentClassLoader);
	        classes.put(className, clazz);
	    }
	    return classes;
	}

	protected Class defineClass(final String className, final Path outputDir,
	        final boolean delegateToParentClassLoader) throws IOException {
		var filename = convertPackageToDirectories(className, ".class");
		var classFile = outputDir.resolve(filename);
		var classData = Files.readAllBytes(classFile);
		return defineClass(className, classData, delegateToParentClassLoader);
	}

    protected Class defineClass(final String className, final byte[] classData, 
            final boolean delegateToParentClassLoader) {
        if(delegateToParentClassLoader) {
		    return new DefiningClassLoader(className, classData).definedClass;
		}
		return new ClassLoaderNotDelegatingToParent(className, classData).definedClass;
    }

	protected String convertPackageToDirectories(final String className, final String fileSuffix) {
		return className.replace('.', File.separatorChar) + fileSuffix;
	}

	protected static class DefiningClassLoader extends ClassLoader {
		
		protected final Class<?> definedClass;
		
		protected DefiningClassLoader(final String className, final byte[] classData) {
			this.definedClass = defineClass(className, classData, 0, classData.length);
		}
	}
	
    protected static class ClassLoaderNotDelegatingToParent extends ClassLoader {
        
        protected final Class<?> definedClass;
        
        protected ClassLoaderNotDelegatingToParent(final String className, final byte[] classData) {
            super(null, null);
            this.definedClass = defineClass(className, classData, 0, classData.length);
        }
    }
    
}
