package de.monochromata.cucumber.stepdefs;

import static java.nio.file.Files.createDirectories;
import static java.nio.file.Files.createTempDirectory;
import static java.nio.file.Files.writeString;
import static org.eclipse.jdt.core.compiler.batch.BatchCompiler.compile;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.Files;
import java.nio.file.Path;

import io.cucumber.docstring.DocString;
import io.cucumber.java.en.Given;
import io.cucumber.java.en.When;

@SuppressWarnings({ "rawtypes", "unchecked" })
public class JavaCompilerStepdefs {

    private final JavaCompilerState state;

    public JavaCompilerStepdefs(final JavaCompilerState state) {
        this.state = state;
    }
    
    @Given("a class {string} from source:")
    public void aClassFromSource(final String className, final DocString javaSource) {
        try {
            var outputDir = compileClass(className, javaSource);
            state.clazz = defineClass(className, outputDir, true);
        } catch (final Exception e) {
            throw new RuntimeException("Failed to compile/load class "+className+", see standard error", e);
        }
    }
    
    @Given("a class {string} from source defined by a class loader that does not delegate to its parent:")
    public void aClassFromSourceOSGI(final String className, final DocString javaSource) {
        try {
            var outputDir = compileClass(className, javaSource);
            state.clazz = defineClass(className, outputDir, false);
        } catch (final Exception e) {
            throw new RuntimeException("Failed to compile/load class "+className+", see standard error", e);
        }
    }
    
    @When("an instance of the class is created")
    public void anInstanceOfTheClassIsCreated() {
        try {
            state.instance = state.clazz.getDeclaredConstructor().newInstance();
        } catch (final Exception e) {
            throw new RuntimeException("Failed to instantiate class via no-args constructor", e);
        }
    }

    protected Path compileClass(final String className, final DocString javaSource) throws IOException {
        var sourceFile = saveSource(className, javaSource.getContent());
        var outputDir = createTempDirectory("outputDir");
        if(!compile("-11 " + sourceFile + " -d "+outputDir, new PrintWriter(System.out), new PrintWriter(System.err), null)) {
            throw new RuntimeException("Compilation failed");
        }
        return outputDir;
    }

	protected Path saveSource(final String className, final String javaSource) throws IOException {
		var inputDir = createTempDirectory("inputDir");
		var filename = convertPackageToDirectories(className, ".java");
		var file = inputDir.resolve(filename);
		createDirectories(file.getParent());
		return writeString(file, javaSource);
	}

	protected Class defineClass(final String className, final Path outputDir,
	        final boolean delegateToParentClassLoader) throws IOException {
		var filename = convertPackageToDirectories(className, ".class");
		var classFile = outputDir.resolve(filename);
		var classData = Files.readAllBytes(classFile);
		return defineClass(className, classData, delegateToParentClassLoader);
	}

    protected Class defineClass(final String className, final byte[] classData, 
            final boolean delegateToParentClassLoader) {
        if(delegateToParentClassLoader) {
		    return new DefiningClassLoader(className, classData).definedClass;
		}
		return new ClassLoaderNotDelegatingToParent(className, classData).definedClass;
    }

	protected String convertPackageToDirectories(final String className, final String fileSuffix) {
		return className.replace('.', File.separatorChar) + fileSuffix;
	}

	protected static class DefiningClassLoader extends ClassLoader {
		
		protected final Class<?> definedClass;
		
		protected DefiningClassLoader(final String className, final byte[] classData) {
			this.definedClass = defineClass(className, classData, 0, classData.length);
		}
	}
	
    protected static class ClassLoaderNotDelegatingToParent extends ClassLoader {
        
        protected final Class<?> definedClass;
        
        protected ClassLoaderNotDelegatingToParent(final String className, final byte[] classData) {
            super(null, null);
            this.definedClass = defineClass(className, classData, 0, classData.length);
        }
    }
    
}
