package de.monochromata.cucumber.stepdefs;

import static java.nio.file.Files.createDirectories;
import static java.nio.file.Files.createTempDirectory;
import static java.nio.file.Files.writeString;
import static org.eclipse.jdt.core.compiler.batch.BatchCompiler.compile;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.Files;
import java.nio.file.Path;

import io.cucumber.docstring.DocString;
import io.cucumber.java8.En;

@SuppressWarnings({ "rawtypes", "unchecked" })
public class JavaCompilerStepdefs implements En {

    private final JavaCompilerState state;

    public JavaCompilerStepdefs(final JavaCompilerState state) {
        this.state = state;
        initialize();
    }

    private void initialize() {
        Given("a class {string} from source:", (final String className, final DocString javaSource) -> {
            try {
            	var sourceFile = saveSource(className, javaSource.getContent());
            	var outputDir = createTempDirectory("outputDir");
            	if(!compile("-11 " + sourceFile + " -d "+outputDir, new PrintWriter(System.out), new PrintWriter(System.err), null)) {
            		throw new RuntimeException("Compilation failed");
            	}
            	state.clazz = defineClass(className, outputDir);
            } catch (final Exception e) {
                throw new RuntimeException("Failed to compile/load class "+className+", see standard error", e);
            }
        });

        When("an instance of the class is created", () -> {
            try {
                state.instance = state.clazz.getDeclaredConstructor().newInstance();
            } catch (final Exception e) {
                throw new RuntimeException("Failed to instantiate class via no-args constructor", e);
            }
        });
    }

	protected Path saveSource(final String className, final String javaSource) throws IOException {
		var inputDir = createTempDirectory("inputDir");
		var filename = convertPackageToDirectories(className, ".java");
		var file = inputDir.resolve(filename);
		createDirectories(file.getParent());
		return writeString(file, javaSource);
	}

	protected Class defineClass(final String className, final Path outputDir) throws IOException {
		var filename = convertPackageToDirectories(className, ".class");
		var classFile = outputDir.resolve(filename);
		var classData = Files.readAllBytes(classFile);
		return new DefiningClassLoader(className, classData).definedClass;
	}

	protected String convertPackageToDirectories(final String className, final String fileSuffix) {
		return className.replace('.', File.separatorChar) + fileSuffix;
	}

	protected static class DefiningClassLoader extends ClassLoader {
		
		protected final Class<?> definedClass;
		
		protected DefiningClassLoader(final String className, final byte[] classData) {
			this.definedClass = defineClass(className, classData, 0, classData.length);
		}
		
	}
}
