package de.monochromata.cucumber.report.config;

import static java.lang.Boolean.parseBoolean;
import static java.util.Optional.ofNullable;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UncheckedIOException;
import java.nio.file.Paths;
import java.util.Enumeration;
import java.util.Properties;

import net.masterthought.cucumber.Configuration;

public class ConfigurationFactory {

    private static final String DEFAULT_FILENAME = "cucumber-reporting.properties";
    private static final String CLASSIFICATIONS_PREFIX = "classifications.";
    public static final String CONFIG_FILE_PROPERTY = "cucumber.reporting.config.file";

    public static Configuration getConfiguration(final File outputDir) {
        final Properties properties = loadProperties();
        final String projectName = properties.getProperty("projectName",
                "No Name (add projectName to cucumber-reporting.properties)");
        final Configuration configuration = new Configuration(outputDir, projectName);
        configureParallelTesting(configuration, properties);
        configureRunWithJenkins(configuration, properties);
        configureBuildNumber(configuration, properties);
        configureClassifications(configuration, properties);
        return configuration;
    }

    protected static void configureParallelTesting(final Configuration configuration, final Properties properties) {
        configuration.setParallelTesting(parseBoolean(properties.getProperty("parallelTesting", "false")));
    }

    protected static void configureRunWithJenkins(final Configuration configuration, final Properties properties) {
        configuration.setRunWithJenkins(parseBoolean(properties.getProperty("runWithJenkins", "false")));
    }

    protected static void configureBuildNumber(final Configuration configuration, final Properties properties) {
        configuration.setBuildNumber(properties.getProperty("buildNumber"));
    }

    protected static void configureClassifications(final Configuration configuration, final Properties properties) {
        final Enumeration<Object> keys = properties.keys();
        while (keys.hasMoreElements()) {
            final String qualifiedKey = (String) keys.nextElement();
            if (qualifiedKey.startsWith(CLASSIFICATIONS_PREFIX)) {
                final String key = qualifiedKey.substring(CLASSIFICATIONS_PREFIX.length());
                configuration.addClassifications(key, properties.getProperty(qualifiedKey));
            }
        }
    }

    protected static Properties loadProperties() {
        final Properties properties = new Properties();
        final InputStream stream = getPropertiesStream();
        if (stream != null) {
            try {
                properties.load(new InputStreamReader(stream, "UTF-8"));
            } catch (final IOException e) {
                throw new UncheckedIOException(e);
            }
        }
        return properties;
    }

    protected static InputStream getPropertiesStream() {
        final String filename = getPropertiesFilename();
        final File propertiesFile = ofNullable(Paths.get(filename).toFile())
                .filter(File::exists)
                .filter(File::isFile)
                .filter(File::canRead)
                .orElse(null);
        if (propertiesFile == null && !filename.equals(DEFAULT_FILENAME)) {
            throw new UncheckedIOException(
                    new FileNotFoundException("Cucumber reporting properties file " + filename + " was not found"));
        } else if (propertiesFile != null) {
            try {
                return new FileInputStream(propertiesFile);
            } catch (final FileNotFoundException e) {
                throw new UncheckedIOException(e);
            }
        }
        return null;
    }

    protected static String getPropertiesFilename() {
        return System.getProperty(CONFIG_FILE_PROPERTY, DEFAULT_FILENAME);
    }

}
