package de.monochromata.anaphors.cog;

import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Stream.empty;

import java.util.Collections;
import java.util.List;
import java.util.stream.Stream;

import de.monochromata.anaphors.ast.ASTBasedAnaphorResolution;
import de.monochromata.anaphors.ast.ASTBasedAnaphora;
import de.monochromata.anaphors.ast.relatedexp.RelatedExpression;
import de.monochromata.anaphors.ast.spi.TransformationsSpi;
import de.monochromata.anaphors.cog.memory.Chunk;
import de.monochromata.anaphors.cog.memory.Memory;
import de.monochromata.anaphors.cog.spi.CogSpi;
import de.monochromata.anaphors.cog.transform.AddParameterToCallChain;
import de.monochromata.anaphors.cog.transform.CheckResult;
import de.monochromata.anaphors.cog.transform.NoPreparationRequired;
import de.monochromata.anaphors.cog.transform.PreparatoryTransformation;

/**
 * Resolve the referents of anaphors based on a cognitive model.
 *
 * @param <N>  The node type in the AST
 * @param <E>  The expression type
 * @param <T>  The type type
 * @param <B>  The binding type
 * @param <TB> The type binding type
 * @param <S>  The scope type (optional)
 * @param <I>  The type used to represent identifiers
 * @param <QI> The type used to represent qualified identifiers
 * @param <R>  The sub-type of related expression to use
 * @param <A>  The sub-type of AST-based anaphora to use
 */
public class ActivationBasedAnaphorResolution<N, E, T, B, TB extends B, S, I, QI, R extends RelatedExpression<N, T, B, TB, S, QI, R>, A extends ASTBasedAnaphora<N, E, T, B, TB, S, I, QI, R, A>> {

    private final List<PreparatoryTransformation<N, E, T, B, TB, S, I, QI, R, A>> preparatoryTransformations;
    private final ASTBasedAnaphorResolution<N, E, T, B, TB, S, I, QI, R, A> delegate;

    /**
     * Used in contract testing.
     */
    @SuppressWarnings("unused")
    protected ActivationBasedAnaphorResolution() {
        delegate = null;
        preparatoryTransformations = null;
    }

    public ActivationBasedAnaphorResolution(final ASTBasedAnaphorResolution<N, E, T, B, TB, S, I, QI, R, A> delegate,
            final TransformationsSpi<N, E, T, B, TB, S, I, QI, R, A> transformationsSpi) {
        this.delegate = delegate;
        this.preparatoryTransformations = asList(new NoPreparationRequired<>(transformationsSpi),
                new AddParameterToCallChain<>(transformationsSpi));
    }

    /**
     * Perform anaphora resolution on the AST implementation configured via the
     * service provider interfaces.
     * <p>
     * TODO: Evaluate whether anaphora relations can be transformed into AST-based
     * anaphora relations
     * <p>
     * TODO: Provide means to transform these anaphora relations into AST-based
     * anaphora relations by performing complex AST manipulations / refactorings
     * like introducing parameters to multiple methods ...
     * <ol>
     * <li>Find ways to create Anaphora relations
     * <li>This might include a preparation step that performs complex AST
     * transformations that prepare the referentialization of the anaphor
     * </ol>
     *
     * @param anaphor                  The anaphor that is to be (re-)resolved.
     * @param definiteExpression       The expression that may function as anaphor
     *                                 in the anaphora relation to be generated by
     *                                 this method. If the anaphora relation is to
     *                                 be re-resolved, this can be a non-trivial
     *                                 expression. If the anaphora relation is to be
     *                                 resolved for the first time, this is
     *                                 typically a simple name and might as well be
     *                                 called a definite expression at this point.
     * @param scope                    May be null if not required by the AST
     *                                 implementation configured via the service
     *                                 provider interfaces.
     * @param timestamp                the point in time at which the memory access
     *                                 occurs (in the format returned by
     *                                 {@link System#currentTimeMillis()}
     * @param numberOfChunksToConsider The maximum number of chunks to consider as
     *                                 potential related expressions.
     * @return TODO: Rework: A list of anaphors that the given definite expression
     *         can function as. The list is empty is the anaphor could not be
     *         resolved and will contain more than one element if the anaphor is
     *         ambiguous.
     */
    public List<Resolution<N, E, T, B, TB, S, I, QI, R, A>> resolveAnaphor(final String anaphor,
            final E definiteExpression, final S scope, final long timestamp, final int numberOfChunksToConsider) {
        final Memory<N> memory = CogSpi.getMemory();
        if (memory == null) {
            return Collections.emptyList();
        }
        return memory.getChunks(timestamp, numberOfChunksToConsider).stream()
                .flatMap(chunk -> resolve(chunk, anaphor, definiteExpression, scope)).collect(toList());
    }

    protected Stream<Resolution<N, E, T, B, TB, S, I, QI, R, A>> resolve(final Chunk<N> chunk, final String anaphor,
            final E definiteExpression, final S scope) {
        for (final PreparatoryTransformation<N, E, T, B, TB, S, I, QI, R, A> preparatoryTransformation : preparatoryTransformations) {
            final CheckResult<N, E, S> result = preparatoryTransformation.canPerform(chunk, definiteExpression, scope);
            if (result.canPerformTransformation()) {
                @SuppressWarnings("unchecked")
                final R potentialRelatedExpression = delegate.getRelatedExpression(result.getChunk().getRepresented());
                if (potentialRelatedExpression != null) {
                    // TODO: Die Prüfung, ob AST-basierte Anaphern anwendbar
                    // sind, muss anhand des Chunks als potential related
                    // expression erfolgen und darf dann nicht reachability
                    // involvieren ...
                    return delegate.resolveAnaphor(potentialRelatedExpression, anaphor, definiteExpression, scope)
                            .stream().map(anaphora -> createResolution(result, preparatoryTransformation, anaphora));
                }
            }
        }
        return empty();
    }

    protected Resolution<N, E, T, B, TB, S, I, QI, R, A> createResolution(final CheckResult<N, E, S> checkResult,
            final PreparatoryTransformation<N, E, T, B, TB, S, I, QI, R, A> preparatoryTransformation,
            final A preliminaryAnaphora) {
        return new DefaultResolution<>(checkResult, preparatoryTransformation, preliminaryAnaphora);
    }

}
