package de.monochromata.anaphors.ast.strategy;

import java.util.List;
import java.util.Optional;

import de.monochromata.Strategy;
import de.monochromata.anaphors.ast.ASTBasedAnaphora;
import de.monochromata.anaphors.ast.AnaphorPart;
import de.monochromata.anaphors.ast.RelatedExpressionPart;
import de.monochromata.anaphors.ast.reference.Referent;
import de.monochromata.anaphors.ast.reference.strategy.ReferentializationStrategy;
import de.monochromata.anaphors.ast.relatedexp.RelatedExpression;
import de.monochromata.anaphors.ast.relatedexp.strategy.LocalTempVariableContents;
import de.monochromata.anaphors.ast.relatedexp.strategy.LocalTempVariableIntroducingStrategy;
import de.monochromata.anaphors.ast.relatedexp.strategy.RelatedExpressionStrategy;
import de.monochromata.anaphors.perspectivation.Perspectivation;

/**
 * A strategy for resolving or constructing the referents of anaphors.
 *
 * @param <N>  The node type in the AST
 * @param <E>  The expression type
 * @param <T>  The type type
 * @param <B>  The binding type
 * @param <TB> The type binding type
 * @param <S>  The scope type (optional)
 * @param <I>  The type used to represent identifiers
 * @param <QI> The type used to represent qualified identifiers
 * @param <R>  The sub-type of related expression to use
 * @param <A>  The sub-type of AST-based anaphora to use
 */
public interface AnaphorResolutionStrategy<N, E, T, B, TB extends B, S, I, QI, R extends RelatedExpression<N, T, B, TB, S, QI, R>, A extends ASTBasedAnaphora<N, E, T, B, TB, S, I, QI, R, A>>
        extends Strategy {

    /**
     * Used to decide whether this strategy can create potential anaphora relations
     * for the given potential related expression.
     */
    public boolean canRelateTo(
            final RelatedExpressionStrategy<N, T, B, TB, S, QI, R> potentialRelatedExpressionStrategy);

    /**
     * Generates a list of potential {@link ASTBasedAnaphora} relations. Invokes
     * {@link #canRelateTo(RelatedExpressionStrategy)} and
     * {@link #createAnaphora(Object, String, Object, RelatedExpression, Referent, ReferentializationStrategy)}
     * .
     *
     * @param scope                       The scope containing the anaphor if used
     *                                    by the compiler-specific implementation.
     *                                    Implementations of this method must not
     *                                    access the scope but merely pass it on to
     *                                    SPI's they invoke.
     * @param anaphor                     The anaphor that is to be (re-)resolved.
     * @param anaphorExpression           The expression that may function as
     *                                    anaphor in the anaphora relation to be
     *                                    generated by this method. If the anaphora
     *                                    relation is to be re-resolved, this can be
     *                                    a non-trivial expression. If the anaphora
     *                                    relation is to be resolved for the first
     *                                    time, this is typically a simple name and
     *                                    might as well be called a definite
     *                                    expression at this point.
     * @param potentialRelatedExpressions Related expressions that could potentially
     *                                    constitute an anaphora relation with the
     *                                    given anaphor.
     * @param refStrategies               Referentialisation strategies to be used
     *                                    to generate anaphora relations.
     * @return A list of potential anaphora relations.
     * @see #canRelateTo(RelatedExpressionStrategy)
     * @see #createAnaphora(Object, String, Object, RelatedExpression, Referent,
     *      ReferentializationStrategy)
     */
    public List<A> generatePotentialAnaphora(S scope, String anaphor, E anaphorExpression,
            List<R> potentialRelatedExpressions, List<ReferentializationStrategy<E, TB, S, I, QI>> refStrategies);

    /**
     * Create a single potential {@link ASTBasedAnaphora} relation for a pair of
     * anaphor (TODO: definite expression) and a potential related expression.
     *
     * @param scope                      The scope containing the anaphor if used by
     *                                   the compiler-specific implementation.
     *                                   Implementations of this method must not
     *                                   access the scope but merely pass it on to
     *                                   SPI's they invoke.
     * @param anaphor                    The anaphor that is to be (re-)resolved.
     * @param anaphorExpression          The expression that may function as anaphor
     *                                   in the anaphora relation to be generated by
     *                                   this method. If the anaphora relation is to
     *                                   be re-resolved, this can be a non-trivial
     *                                   expression. If the anaphora relation is to
     *                                   be resolved for the first time, this is
     *                                   typically a simple name and might as well
     *                                   be called a definite expression at this
     *                                   point.
     * @param potentialRelatedExpression A related expression that could potentially
     *                                   constitute an anaphora relation with the
     *                                   given anaphor.
     * @param potentialReferent          As e.g. returned by
     *                                   {@link AbstractAnaphorResolutionStrategy#createPotentialReferents(Object, RelatedExpression)}
     *                                   of this instance.
     * @param refStrategy                The referentialization strategies to be
     *                                   used to generate the anaphora relation.
     * @return A list of potential anaphora relations.
     */
    public A createAnaphora(S scope, String anaphor, E anaphorExpression, R potentialRelatedExpression,
            Referent<TB, S, I, QI> potentialReferent, ReferentializationStrategy<E, TB, S, I, QI> refStrategy);

    /**
     * Re-creates a referent from a given related expression and memento.
     * <p>
     * TODO: This method should be renamed to reCreateReferent(...)
     * <p>
     * TODO: When should this method be used?
     *
     * @param scope             The scope containing the anaphor if used by the
     *                          compiler-specific implementation. Implementations of
     *                          this method must not access the scope but merely
     *                          pass it on to SPI's they invoke.
     * @param relatedExpression Related expressions that could potentially
     *                          constitute an anaphora relation with the given
     *                          anaphor.
     * @param memento           A compiler-specific memento used to reconstruct the
     *                          referent.
     * @return The re-constructed referent.
     * @throws IllegalArgumentException If the referent cannot be re-created from
     *                                  the given related expression and memento
     * @see Referent#getMemento()
     */
    public Referent<TB, S, I, QI> createReferent(S scope, R relatedExpression, Object memento);

    /**
     * Replaces the anaphor of the given anaphora relation.
     *
     * Note that this method will replace AST nodes in the given anaphora instance
     * if realization involves replacing AST nodes.
     *
     * @param replacee        The anaphor node that will be replaced. TODO: Is this
     *                        the anaphor, really?
     * @param guessedTempName If the given related expression part has a
     *                        {@link LocalTempVariableIntroducingStrategy}, the
     *                        strategy introduced a temporary variable that might be
     *                        used by the anaphor resolution strategy to generate
     *                        code for the anaphors.
     * @param support         Optional compiler-specific support objects.
     *                        Implementations of this method must not access the
     *                        support objects but merely pass them on to SPI's they
     *                        invoke.
     * @return The node that replaced the anaphor.
     * @see ASTBasedAnaphora#getAnaphor()
     */
    public E realize(RelatedExpressionPart<N, E, T, B, TB, S, I, QI, R> relatedExpressionPart,
            AnaphorPart<N, E, T, B, TB, S, I, QI, R, A> anaphorPart, E replacee, Optional<I> guessedTempName,
            Object... support);

    public List<Perspectivation> underspecifyAnaphor(final R relatedExpression, final String anaphor,
            final E anaphorExpression, final Referent<TB, S, I, QI> referent, final S scope);

    LocalTempVariableContents getLocalTempVariableContents();

    /**
     * To be invoked when a related expression resolved with this strategy is
     * realized to obtain the kind of anaphor resolution strategy that applies after
     * realization.
     * <p>
     * That realized kind will also apply when the anaphora is re-resolved. Thus
     * setting the realized kind in the beginning will avoid a faux change being
     * detected during re-resolution of the unmodified anaphora relation.
     */
    String getKindOfAnaphorResolutionStrategyToBeRealized(
            final AnaphorPart<N, E, T, B, TB, S, I, QI, R, A> anaphorPart);

    /**
     * To be invoked when an anaphora resolved with this strategy is realized to
     * obtain the kind of referentialization that applies after realization.
     * <p>
     * That realized kind will also apply when the anaphora is re-resolved. Thus
     * setting the realized kind in the beginning will avoid a faux change being
     * detected during re-resolution of the unmodified anaphora relation.
     */
    String getKindOfReferentializationStrategyToBeRealized(
            final AnaphorPart<N, E, T, B, TB, S, I, QI, R, A> anaphorPart);

    /**
     * To be invoked when a related expression resolved with this strategy is
     * realized to obtain the anaphor that will be present after realization.
     * <p>
     * That realized anaphor will also apply when the anaphora is re-resolved. Thus
     * setting the realized anaphor in the beginning will avoid a faux change being
     * detected during re-resolution of the unmodified anaphora relation.
     */
    String getAnaphorToBeRealized(final RelatedExpressionPart<N, E, T, B, TB, S, I, QI, R> relatedExpressionPart,
            final List<AnaphorPart<N, E, T, B, TB, S, I, QI, R, A>> allAnaphorPartsRelatedToTheRelatedExpression,
            final AnaphorPart<N, E, T, B, TB, S, I, QI, R, A> anaphorPart, final S scope);
}
