package de.mklinger.qetcher.htmlinliner;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URLConnection;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.Collection;
import java.util.Locale;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.io.IOUtils;
import org.apache.commons.io.output.WriterOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class ResourceLoaderImpl implements ResourceLoader {
	private static final Logger LOG = LoggerFactory.getLogger(ResourceLoaderImpl.class);

	private static final Pattern CHARSET_PATTERN = Pattern.compile(";\\s*charset=([^;]+)", Pattern.CASE_INSENSITIVE);

	private final Supplier<Collection<Cookie>> cookieSupplier;

	public ResourceLoaderImpl(final Supplier<Collection<Cookie>> cookieSupplier) {
		this.cookieSupplier = cookieSupplier;
	}

	@Override
	public Optional<String> getContents(final URI url, final URI referrer) {
		LOG.info("Fetching {} for inline contents", url);
		try {
			final URLConnection connection = get(url, referrer);

			final Charset charset = getCharset(connection);
			final StringWriter sw = new StringWriter();
			final WriterOutputStream wout = new WriterOutputStream(sw, charset);
			try (InputStream in = connection.getInputStream()) {
				IOUtils.copy(in, wout);
			}
			wout.flush();
			final String contents = sw.toString();
			return Optional.of(contents);
		} catch (final Exception e) {
			// no stack trace here.
			LOG.info("Error getting contents for inlining: {}", url);
			LOG.debug("Error getting contents for inlining had exception:", e);
			return Optional.empty();
		}
	}

	@Override
	public Optional<String> getExternalInlineImgSrc(final URI url, final URI referrer) {
		LOG.info("Fetching {} for inline image", url);
		try {
			final URLConnection connection = get(url, referrer);
			final String contentType = getImageContentType(connection);
			try (InputStream in = connection.getInputStream()) {
				return Optional.of(getInlineImgSrc(contentType, in));
			}
		} catch (final Exception e) {
			// no stack trace here.
			LOG.info("Error getting contents for inlining: {}", url);
			LOG.debug("Error getting contents for inlining had exception:", e);
			return Optional.empty();
		}
	}

	@Override
	public String getInlineImgSrc(final String contentType, final InputStream in) throws IOException {
		final byte[] data = IOUtils.toByteArray(in);
		final String base64Data = Base64.getEncoder().encodeToString(data);

		final StringBuilder sb = new StringBuilder(base64Data.length() + "data:".length() + contentType.length() + ";base64,".length());
		sb.append("data:");
		sb.append(contentType);
		sb.append(";base64,");
		sb.append(base64Data);

		return sb.toString();
	}

	private URLConnection get(final URI uri, final URI referrer) throws IOException {
		final URLConnection connection = uri.toURL().openConnection();
		if (isSameHost(uri, referrer)) {
			LOG.debug("Using cookies for url {}", uri);
			setCookies(connection, cookieSupplier);
		}

		connection.connect();

		if (connection instanceof HttpURLConnection) {
			final int statusCode = ((HttpURLConnection)connection).getResponseCode();
			if (statusCode != 200) {
				throw new HtmlElementInlinerException("Non 200 status code (" + statusCode + ") for content: " + uri);
			}
		}
		return connection;
	}

	private boolean isSameHost(final URI uri1, final URI uri2) {
		return
				uri1.getScheme() != null && uri1.getScheme().equals(uri2.getScheme()) &&
				uri1.getHost() != null && uri1.getHost().equals(uri2.getHost()) &&
				uri1.getPort() == uri2.getPort();
	}

	private void setCookies(final URLConnection connection, final Supplier<Collection<Cookie>> cookieSupplier) {
		if (cookieSupplier == null) {
			LOG.debug("No cookie supplier available");
			return;
		}
		final Collection<Cookie> cookies = cookieSupplier.get();
		if (cookies == null || cookies.isEmpty()) {
			LOG.debug("No cookies");
			return;
		}
		for (final Cookie cookie : cookies) {
			// XXX escape cookie name/value ??
			// Cookie names and values seem not to be escapable. Only certain
			// characters are allowed for name and for value.
			// For value see https://tools.ietf.org/html/rfc6265.html.
			// Should we validate here and ignore invalid cookies?
			connection.addRequestProperty("Cookie", cookie.getName() + "=" + cookie.getValue());
		}
	}

	private Charset getCharset(final URLConnection connection) {
		final String contentType = connection.getContentType();
		if (contentType != null && !contentType.isEmpty()) {
			final Matcher matcher = CHARSET_PATTERN.matcher(contentType);
			if (matcher.find()) {
				final String responseCharset = matcher.group(1).trim();
				try {
					final Charset charset = Charset.forName(responseCharset);
					LOG.debug("Using response charset: {}", charset);
					return charset;
				} catch (final Exception e) {
					// ignore
				}
			}
		}
		return StandardCharsets.UTF_8;
	}

	private String getImageContentType(final URLConnection connection) {
		final String contentType = connection.getContentType();
		if (contentType == null) {
			throw new HtmlElementInlinerException("No content type available for inline image");
		}
		if (!contentType.toLowerCase(Locale.US).startsWith("image/")) {
			throw new HtmlElementInlinerException("Unsupported content type for inline image: '" + contentType + "'");
		}
		return contentType;
	}

	@Override
	public void close() {
		// Do nothing
	}
}
