package de.mklinger.qetcher.client.model.v1.builder;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import de.mklinger.qetcher.client.common.MediaType;
import de.mklinger.qetcher.client.model.v1.ConversionFile;
import de.mklinger.qetcher.client.model.v1.Error;
import de.mklinger.qetcher.client.model.v1.Job;
import de.mklinger.qetcher.client.model.v1.JobState;
import de.mklinger.qetcher.client.model.v1.impl.JobImpl;

/**
 * Created by Marius Heinzmann on 3/26/18.
 *
 * @author Marius Heinzmann - mheinzmann[at]mklinger[dot]de
 */
public class JobBuilder {
	private String jobId;
	private String parentJobId;
	private String tenantId;
	private String nodeId;
	private String converterId;
	private int priority;
	private MediaType fromMediaType;
	private List<String> inputFileIds;
	private MediaType toMediaType;
	private List<String> resultFileIds;
	private JobState state;
	private Error error;
	private Map<String, String> properties;
	private Instant created;
	private Instant updated;
	private Duration estimatedRunTime;
	private String referrer;
	private Duration cancelTimeout;
	private Duration deleteTimeout;

	public static JobBuilder of(final Job job) {
		return new JobBuilder()
				.jobId(job.getJobId())
				.parentJobId(job.getParentJobId())
				.tenantId(job.getTenantId())
				.nodeId(job.getNodeId())
				.converterId(job.getConverterId())
				.priority(job.getPriority())
				.fromMediaType(job.getFromMediaType())
				.inputFileIds(job.getInputFileIds())
				.toMediaType(job.getToMediaType())
				.resultFileIds(job.getResultFileIds())
				.error(job.getError())
				.state(job.getState())
				.properties(job.getProperties())
				.created(job.getCreated())
				.updated(Instant.now())
				.estimatedRunTime(job.getEstimatedRunTime())
				.referrer(job.getReferrer())
				.cancelTimeout(job.getCancelTimeout())
				.deleteTimeout(job.getDeleteTimeout());
	}

	public JobBuilder jobId(final String jobId) {
		this.jobId = jobId;
		return this;
	}

	public JobBuilder parentJobId(final String parentJobId) {
		this.parentJobId = parentJobId;
		return this;
	}

	public JobBuilder tenantId(final String tenantId) {
		this.tenantId = tenantId;
		return this;
	}

	public JobBuilder nodeId(final String nodeId) {
		this.nodeId = nodeId;
		return this;
	}

	public JobBuilder converterId(final String converterId) {
		this.converterId = converterId;
		return this;
	}

	/**
	 * Set the job priority. Higher value is higher priority. Jobs with higher
	 * priorities will be processed before jobs with lower priority.
	 */
	public JobBuilder priority(final int priority) {
		this.priority = priority;
		return this;
	}

	public JobBuilder fromMediaType(final MediaType fromMediaType) {
		this.fromMediaType = fromMediaType;
		return this;
	}

	public JobBuilder inputFiles(final List<ConversionFile> inputFiles) {
		if (inputFiles == null || inputFiles.isEmpty()) {
			return inputFileIds(null);
		} else {
			if (inputFileIds == null || inputFileIds.isEmpty()) {
				fromMediaType(inputFiles.get(0).getMediaType());
			}
			return inputFileIds(inputFiles.stream()
					.map(ConversionFile::getFileId)
					.collect(Collectors.toList()));
		}
	}

	public JobBuilder inputFileIds(final List<String> inputFileIds) {
		if (inputFileIds == null || inputFileIds.isEmpty()) {
			this.inputFileIds = null;
		} else {
			this.inputFileIds = new ArrayList<>(inputFileIds);
		}
		return this;
	}

	public JobBuilder withInputFile(final ConversionFile inputFile) {
		Objects.requireNonNull(inputFile);
		if (inputFileIds == null || inputFileIds.isEmpty()) {
			fromMediaType(inputFile.getMediaType());
		}
		return withInputFileId(inputFile.getFileId());
	}

	public JobBuilder withInputFileId(final String inputFileId) {
		if (inputFileIds == null) {
			return inputFileIds(Collections.singletonList(inputFileId));
		} else if (!inputFileIds.contains(inputFileId)) {
			inputFileIds.add(inputFileId);
		}
		return this;
	}

	public JobBuilder toMediaType(final MediaType toMediaType) {
		this.toMediaType = toMediaType;
		return this;
	}

	public JobBuilder resultFiles(final List<ConversionFile> resultFiles) {
		if (resultFiles == null || resultFiles.isEmpty()) {
			this.resultFileIds = null;
			return this;
		} else {
			return resultFileIds(resultFiles.stream()
					.map(ConversionFile::getFileId)
					.collect(Collectors.toList()));
		}
	}

	public JobBuilder resultFileIds(final List<String> resultFileIds) {
		if (resultFileIds == null || resultFileIds.isEmpty()) {
			this.resultFileIds = null;
		} else {
			this.resultFileIds = new ArrayList<>(resultFileIds);
		}
		return this;
	}

	public JobBuilder withResultFileId(final String resultFileId) {
		if (resultFileIds == null) {
			resultFileIds = new ArrayList<>();
			resultFileIds.add(resultFileId);
		} else if (!resultFileIds.contains(resultFileId)) {
			resultFileIds.add(resultFileId);
		}
		return this;
	}

	public JobBuilder state(final JobState state) {
		this.state = state;
		return this;
	}

	/**
	 * This also sets the job's state to {@link JobState#ERROR}.
	 */
	public JobBuilder error(final Error error) {
		if (error != null) {
			state(JobState.ERROR);
		}
		this.error = error;
		return this;
	}

	public JobBuilder properties(final Map<String, String> properties) {
		if (properties == null) {
			this.properties = null;
		} else {
			this.properties = new HashMap<>(properties);
		}
		return this;
	}

	public JobBuilder withProperty(final String name, final String value) {
		if (properties == null) {
			properties = new HashMap<>();
		}
		properties.put(name, value);
		return this;
	}

	public JobBuilder created(final Instant created) {
		this.created = created;
		return this;
	}

	public JobBuilder updated(final Instant updated) {
		this.updated = updated;
		return this;
	}

	public JobBuilder estimatedRunTime(final Duration estimatedRunTime) {
		this.estimatedRunTime = estimatedRunTime;
		return this;
	}

	public JobBuilder referrer(final String referrer) {
		this.referrer = referrer;
		return this;
	}

	public JobBuilder cancelTimeout(final Duration cancelTimeout) {
		this.cancelTimeout = cancelTimeout;
		return this;
	}

	public JobBuilder deleteTimeout(final Duration deleteTimeout) {
		this.deleteTimeout = deleteTimeout;
		return this;
	}

	public Job build() {
		return new JobImpl(this);
	}

	public String getJobId() {
		return jobId;
	}

	public String getParentJobId() {
		return parentJobId;
	}

	public String getTenantId() {
		return tenantId;
	}

	public String getNodeId() {
		return nodeId;
	}

	public String getConverterId() {
		return converterId;
	}

	/**
	 * Get the job priority. Higher value is higher priority. Jobs with higher
	 * priorities will be processed before jobs with lower priority.
	 */
	public int getPriority() {
		return priority;
	}

	public MediaType getFromMediaType() {
		return fromMediaType;
	}

	public List<String> getInputFileIds() {
		return inputFileIds;
	}

	public MediaType getToMediaType() {
		return toMediaType;
	}

	public List<String> getResultFileIds() {
		return resultFileIds;
	}

	public JobState getState() {
		return state;
	}

	public Error getError() {
		return error;
	}

	public Map<String, String> getProperties() {
		return properties;
	}

	public Instant getCreated() {
		return created;
	}

	public Instant getUpdated() {
		return updated;
	}

	public Duration getEstimatedRunTime() {
		return estimatedRunTime;
	}

	public String getReferrer() {
		return referrer;
	}

	public Duration getCancelTimeout() {
		return cancelTimeout;
	}

	public Duration getDeleteTimeout() {
		return deleteTimeout;
	}
}
