package de.mklinger.qetcher.client.common;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class MediaType {
	private static final String MEDIA_TYPE_WILDCARD = "*";
	private final String type;
	private final String subtype;
	private Map<String, String> parameters;

	/**
	 * Creates a new instance of {@code MediaType} with the supplied type, subtype and
	 * parameters.
	 *
	 * @param type       the primary type, {@code null} is equivalent to
	 *                   {@link #MEDIA_TYPE_WILDCARD}.
	 * @param subtype    the subtype, {@code null} is equivalent to
	 *                   {@link #MEDIA_TYPE_WILDCARD}.
	 * @param parameters a map of media type parameters, {@code null} is the same as an
	 *                   empty map.
	 */
	public MediaType(final String type, final String subtype, final Map<String, String> parameters) {
		this.type = type == null ? MEDIA_TYPE_WILDCARD : type;
		this.subtype = subtype == null ? MEDIA_TYPE_WILDCARD : subtype;

		if (parameters == null) {
			this.parameters = Collections.emptyMap();
		} else {
			this.parameters = Collections.unmodifiableMap(parameters);
		}
	}

	/**
	 * Creates a new instance of {@code MediaType} with the supplied type and subtype.
	 *
	 * @param type    the primary type, {@code null} is equivalent to
	 *                {@link #MEDIA_TYPE_WILDCARD}
	 * @param subtype the subtype, {@code null} is equivalent to
	 *                {@link #MEDIA_TYPE_WILDCARD}
	 */
	public MediaType(final String type, final String subtype) {
		this(type, subtype,  null);
	}

	/**
	 * Creates a new instance of {@code MediaType} by parsing the supplied string.
	 *
	 * @param type the media type string.
	 * @return the newly created MediaType.
	 * @throws IllegalArgumentException if the supplied string cannot be parsed
	 *                                  or is {@code null}.
	 */
	public static MediaType valueOf(final String type) {
		return parse(type);
	}


	/**
	 * Getter for primary type.
	 *
	 * @return value of primary type.
	 */
	public String getType() {
		return this.type;
	}

	/**
	 * Checks if the primary type is a wildcard.
	 *
	 * @return true if the primary type is a wildcard.
	 */
	public boolean isWildcardType() {
		return this.getType().equals(MEDIA_TYPE_WILDCARD);
	}

	/**
	 * Getter for subtype.
	 *
	 * @return value of subtype.
	 */
	public String getSubtype() {
		return this.subtype;
	}

	/**
	 * Checks if the subtype is a wildcard.
	 *
	 * @return true if the subtype is a wildcard.
	 */
	public boolean isWildcardSubtype() {
		return this.getSubtype().equals(MEDIA_TYPE_WILDCARD);
	}

	/**
	 * Getter for a read-only parameter map. Keys are case-insensitive.
	 *
	 * @return an immutable map of parameters.
	 */
	public Map<String, String> getParameters() {
		return parameters;
	}

	/**
	 * Get an instance with same type and sub-type but without parameters.
	 */
	public MediaType withoutParameters() {
		if (parameters == null || parameters.isEmpty()) {
			return this;
		}
		return new MediaType(type, subtype);
	}

	/**
	 * Get an instance with same type and sub-type and parameters, but with
	 * given parameters removed.
	 */
	public MediaType withoutParameters(final String... names) {
		// In these cases, we expect the method above being called.
		// If not, maybe rename one of the methods.
		assert names != null;
		assert names.length > 0;

		if (parameters == null || parameters.isEmpty()) {
			return this;
		}
		return new MediaType(type, subtype);
	}

	/**
	 * Get an instance with same type, sub-type and parameters with an
	 * additional parameter as given. If there was already a parameter
	 * with the same name, its value will be replaced by the new value.
	 */
	public MediaType withParameter(final String name, final String value) {
		return withParameters(Collections.singletonMap(name, value));
	}

	/**
	 * Get an instance with same type, sub-type and parameters with
	 * additional parameters as given. If there where already parameters
	 * with the same names, its values will be replaced by the new values.
	 */
	public MediaType withParameters(final Map<String, String> additionalParameters) {
		if (additionalParameters == null || additionalParameters.isEmpty()) {
			return this;
		}
		if (parameters == null || parameters.isEmpty()) {
			return new MediaType(type, subtype, additionalParameters);
		}
		final Map<String, String> allParameters = new HashMap<>(parameters);
		allParameters.putAll(additionalParameters);
		return new MediaType(type, subtype, allParameters);
	}

	/**
	 * Check if this media type is compatible with another media type. E.g.
	 * image/* is compatible with image/jpeg, image/png, etc. Media type
	 * parameters are ignored. The function is commutative.
	 *
	 * @param other the media type to compare with.
	 * @return true if the types are compatible, false otherwise.
	 */
	public boolean isCompatible(final MediaType other) {
		return other != null && // return false if other is null, else
				(type.equals(MEDIA_TYPE_WILDCARD) || other.type.equals(MEDIA_TYPE_WILDCARD) || // both are wildcard types, or
						(type.equalsIgnoreCase(other.type) && (subtype.equals(MEDIA_TYPE_WILDCARD)
								|| other.subtype.equals(MEDIA_TYPE_WILDCARD))) || // same types, wildcard sub-types, or
						(type.equalsIgnoreCase(other.type) && this.subtype.equalsIgnoreCase(other.subtype))); // same types & sub-types
	}

	/**
	 * Compares {@code obj} to this media type to see if they are the same by comparing
	 * type, subtype and parameters. Note that the case-sensitivity of parameter
	 * values is dependent on the semantics of the parameter name, see
	 * {@link <a href="http://www.w3.org/Protocols/rfc2616/rfc2616-sec3.html#sec3.7">HTTP/1.1</a>}.
	 * This method assumes that values are case-sensitive.
	 * <p/>
	 * Note that the {@code equals(...)} implementation does not perform
	 * a class equality check ({@code this.getClass() == obj.getClass()}). Therefore
	 * any class that extends from {@code MediaType} class and needs to override
	 * one of the {@code equals(...)} and {@link #hashCode()} methods must
	 * always override both methods to ensure the contract between
	 * {@link Object#equals(java.lang.Object)} and {@link Object#hashCode()} does
	 * not break.
	 *
	 * @param obj the object to compare to.
	 * @return true if the two media types are the same, false otherwise.
	 */
	@Override
	public boolean equals(final Object obj) {
		if (!(obj instanceof MediaType)) {
			return false;
		}

		final MediaType other = (MediaType) obj;
		return (this.type.equalsIgnoreCase(other.type)
				&& this.subtype.equalsIgnoreCase(other.subtype)
				&& this.parameters.equals(other.parameters));
	}

	/**
	 * Generate a hash code from the type, subtype and parameters.
	 * <p/>
	 * Note that the {@link #equals(java.lang.Object)} implementation does not perform
	 * a class equality check ({@code this.getClass() == obj.getClass()}). Therefore
	 * any class that extends from {@code MediaType} class and needs to override
	 * one of the {@link #equals(Object)} and {@code hashCode()} methods must
	 * always override both methods to ensure the contract between
	 * {@link Object#equals(java.lang.Object)} and {@link Object#hashCode()} does
	 * not break.
	 *
	 * @return a generated hash code.
	 */
	@Override
	public int hashCode() {
		return (this.type.toLowerCase() + this.subtype.toLowerCase()).hashCode() + this.parameters.hashCode();
	}

	/**
	 * Convert the media type to a string suitable for use as the value of a
	 * corresponding HTTP header.
	 *
	 * @return a string version of the media type.
	 */
	@Override
	public String toString() {
		return toString(this, true);
	}

	/**
	 * Convert the media type to a string without parameters.
	 * @return a string version of the media type without parameters.
	 */
	public String getFullType() {
		return toString(this, false);
	}

	private static final char[] quotedChars = "()<>@,;:\\\"/[]?= \t\r\n".toCharArray();

	public static boolean quoted(final String str) {
		for (final char c : str.toCharArray()) {
			for (final char q : quotedChars) {
				if (c == q) {
					return true;
				}
			}
		}
		return false;
	}

	private static String toString(final MediaType type, final boolean withParameters) {
		if (type == null) {
			throw new IllegalArgumentException("param was null");
		}
		final StringBuilder buf = new StringBuilder();

		buf.append(type.getType().toLowerCase());
		buf.append("/");
		buf.append(type.getSubtype().toLowerCase());
		if (!withParameters || type.getParameters() == null || type.getParameters().isEmpty()) {
			return buf.toString();
		}
		for (final String name : type.getParameters().keySet()) {
			buf.append(';');
			buf.append(name);
			buf.append('=');
			final String val = type.getParameters().get(name);
			if (quoted(val)) {
				buf.append('"');
				buf.append(val);
				buf.append('"');
			} else {
				buf.append(val);
			}
		}
		return buf.toString();
	}

	private static MediaType parse(String type) {
		String params = null;
		final int idx = type.indexOf(";");
		if (idx > -1)
		{
			params = type.substring(idx + 1).trim();
			type = type.substring(0, idx);
		}
		String major = null;
		String subtype = null;
		final String[] paths = type.split("/");
		if (paths.length < 2 && type.equals("*"))
		{
			major = "*";
			subtype = "*";

		}
		else if (paths.length != 2
				|| "".equals(paths[0]) || "".equals(paths[1])
				|| paths[0].contains(" ") || paths[1].contains(" "))
		{
			throw new IllegalArgumentException("Failure parsing MediaType string: " + type);
		}
		else if (paths.length == 2)
		{
			major = paths[0];
			subtype = paths[1];
		}
		if (params != null && !params.equals(""))
		{
			final HashMap<String, String> typeParams = new HashMap<>();

			int start = 0;

			while (start < params.length())
			{
				start = setParam(typeParams, params, start);
			}
			return new MediaType(major, subtype, typeParams);
		}
		else
		{
			return new MediaType(major, subtype);
		}
	}

	private static int getEndName(final String params, final int start) {
		final int equals = params.indexOf('=', start);
		final int semicolon = params.indexOf(';', start);
		if (equals == -1 && semicolon == -1) {
			return params.length();
		}
		if (equals == -1) {
			return semicolon;
		}
		if (semicolon == -1) {
			return equals;
		}
		final int end = (equals < semicolon) ? equals : semicolon;
		return end;
	}

	private static int setParam(final HashMap<String, String> typeParams, final String params, final int start) {
		boolean quote = false;
		boolean backslash = false;

		int end = getEndName(params, start);
		final String name = params.substring(start, end).trim();
		if (end < params.length() && params.charAt(end) == '=') {
			end++;
		}

		final StringBuilder buffer = new StringBuilder(params.length() - end);
		int i = end;
		for (; i < params.length(); i++) {
			final char c = params.charAt(i);
			switch (c) {
			case '"':
				if (backslash) {
					backslash = false;
					buffer.append(c);
				} else {
					quote = !quote;
				}
				break;
			case '\\':
				if (backslash) {
					backslash = false;
					buffer.append(c);
				}
				break;
			case ';':
				if (!quote) {
					final String value = buffer.toString().trim();
					typeParams.put(name, value);
					return i + 1;
				} else {
					buffer.append(c);
				}
				break;
			default:
				buffer.append(c);
				break;
			}
		}
		final String value = buffer.toString().trim();
		typeParams.put(name, value);
		return i;
	}

	public String getParameter(final String name) {
		final String value = getParameters().get(name);
		if (value == null || value.isEmpty()) {
			return null;
		}
		return value;
	}

	public Integer getParameterInteger(final String name) {
		final String value = getParameters().get(name);
		if (value == null || value.isEmpty()) {
			return null;
		}
		return Integer.parseInt(value);
	}

	public boolean getParameterBoolean(final String name, final boolean defaultValue) {
		final String value = getParameters().get(name);
		if (value == null || value.isEmpty()) {
			return defaultValue;
		}
		return Boolean.parseBoolean(value);
	}
}
