/*
 * Copyright 2016-present mklinger GmbH - http://www.mklinger.de
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.mklinger.qetcher.client.common;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.util.Objects;

import de.mklinger.qetcher.client.common.annotations.Nullable;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de - klingerm
 */
public class KeyStores {
	private static final String CLASSPATH_PREFIX = "classpath:";

	/** No instantiation */
	private KeyStores() {}

	public static KeyStore load(final String location, @Nullable final String password) {
		return load(location, password, KeyStore.getDefaultType(), getDefaultClassLoader());
	}

	public static KeyStore load(final String location, @Nullable final String password, final ClassLoader classLoader) {
		return load(location, password, KeyStore.getDefaultType(), classLoader);
	}

	public static KeyStore load(final String location, @Nullable final String password, final String type) {
		return load(location, password, type, getDefaultClassLoader());
	}

	private static ClassLoader getDefaultClassLoader() {
		return KeyStores.class.getClassLoader();
	}

	public static KeyStore load(final String location, @Nullable final String password, final String type, final ClassLoader classLoader) {
		Objects.requireNonNull(location);
		Objects.requireNonNull(type);
		Objects.requireNonNull(classLoader);
		try {
			final KeyStore keyStore = KeyStore.getInstance(type);
			try(InputStream in = newInputStream(location, classLoader)) {
				keyStore.load(in, toCharArray(password));
			}
			return keyStore;
		} catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
			throw new QetcherException("Error loading keystore from location '{}'", location, e);
		}
	}

	private static InputStream newInputStream(final String location, final ClassLoader classLoader) throws IOException {
		if (location.startsWith(CLASSPATH_PREFIX)) {
			final String classpathLocation = location.substring(CLASSPATH_PREFIX.length());
			InputStream in = classLoader.getResourceAsStream(classpathLocation);
			if (in == null) {
				in = Thread.currentThread().getContextClassLoader().getResourceAsStream(classpathLocation);
			}
			if (in == null) {
				throw new FileNotFoundException("Classpath resource not found: " + classpathLocation);
			}
			return in;
		} else if (location.startsWith("/") || location.startsWith("./")) {
			return new FileInputStream(location);
		} else {
			return URI.create(location).toURL().openStream();
		}
	}

	private static char[] toCharArray(final String password) {
		if (password == null) {
			return null;
		}
		return password.toCharArray();
	}
}
