package de.mklinger.qetcher.client.impl.retry;

import java.io.File;
import java.io.IOException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import de.mklinger.micro.annotations.VisibleForTesting;
import de.mklinger.micro.throwables.Throwables;
import de.mklinger.qetcher.client.InputConversionFile;
import de.mklinger.qetcher.client.InputJob;
import de.mklinger.qetcher.client.QetcherRemoteException;
import de.mklinger.qetcher.client.common.concurrent.Delay;
import de.mklinger.qetcher.client.impl.QetcherClientBuilderImpl;
import de.mklinger.qetcher.client.impl.QetcherClientImpl;
import de.mklinger.qetcher.client.impl.lookup.ServiceUriSupplier;
import de.mklinger.qetcher.client.model.v1.AvailableConversion;
import de.mklinger.qetcher.client.model.v1.AvailableNode;
import de.mklinger.qetcher.client.model.v1.ConversionFile;
import de.mklinger.qetcher.client.model.v1.Job;

/**
 * Retry on 404 and 417 responses and IOExceptions.
 *
 * This can not be implemented as a pure decorator with delegate on top of
 * QetcherClient interface, as we also have to wrap calls made internally,
 * especially calls initiated by {@link #getJobDone(String)}.
 *
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class RetryingQetcherClientImpl extends QetcherClientImpl {
	private static final Logger LOG = LoggerFactory.getLogger(RetryingQetcherClientImpl.class);

	private static final int MAX_TRIES = 10;
	private static final int DEFAULT_MAX_WAIT_TIME_MILLIS = 1000;
	private final long maxWaitTimeMillis;

	public RetryingQetcherClientImpl(final QetcherClientBuilderImpl builder, final ServiceUriSupplier serviceUriSupplier) {
		this(builder, serviceUriSupplier, DEFAULT_MAX_WAIT_TIME_MILLIS);
	}

	@VisibleForTesting
	protected RetryingQetcherClientImpl(final QetcherClientBuilderImpl builder, final ServiceUriSupplier serviceUriSupplier, final long maxWaitTimeMillis) {
		super(builder, serviceUriSupplier);
		this.maxWaitTimeMillis = maxWaitTimeMillis;
	}

	@Override
	public CompletableFuture<ConversionFile> uploadFile(final InputConversionFile inputFile) {
		return doWithRetry(() -> super.uploadFile(inputFile));
	}

	@Override
	public CompletableFuture<ConversionFile> getFile(final String fileId) {
		return doWithRetry(() -> super.getFile(fileId));
	}

	@Override
	public CompletableFuture<List<ConversionFile>> getFiles() {
		return doWithRetry(() -> super.getFiles());
	}

	@Override
	public CompletableFuture<Void> deleteFile(final String fileId) {
		return doWithRetry(() -> super.deleteFile(fileId));
	}

	@Override
	public CompletableFuture<Path> downloadAsFile(final String fileId, final Path file, final OpenOption... openOptions) {
		return doWithRetry(() -> super.downloadAsFile(fileId, file, openOptions));
	}

	@Override
	public CompletableFuture<Job> createJob(final InputJob inputJob) {
		return doWithRetry(() -> super.createJob(inputJob));
	}

	@Override
	public CompletableFuture<Job> getJob(final String jobId) {
		return doWithRetry(() -> super.getJob(jobId));
	}

	@Override
	public CompletableFuture<List<Job>> getJobs() {
		return doWithRetry(() -> super.getJobs());
	}

	@Override
	public CompletableFuture<Void> deleteJob(final String jobId) {
		return doWithRetry(() -> super.deleteJob(jobId));
	}

	@Override
	public CompletableFuture<List<AvailableConversion>> getAvailableConversions() {
		return doWithRetry(() -> super.getAvailableConversions());
	}

	@Override
	public CompletableFuture<List<AvailableNode>> getAvailableNodes() {
		return doWithRetry(() -> super.getAvailableNodes());
	}

	@Override
	public CompletableFuture<ConversionFile> getFile(final ConversionFile file) {
		return doWithRetry(() -> super.getFile(file));
	}

	@Override
	public CompletableFuture<Void> deleteFile(final ConversionFile file) {
		return doWithRetry(() -> super.deleteFile(file));
	}

	@Override
	public CompletableFuture<Path> downloadAsFile(final String fileId, final Path file) {
		return doWithRetry(() -> super.downloadAsFile(fileId, file));
	}

	@Override
	public CompletableFuture<File> downloadAsFile(final String fileId, final File file) {
		return doWithRetry(() -> super.downloadAsFile(fileId, file));
	}

	@Override
	public CompletableFuture<Path> downloadAsTempFile(final String fileId) {
		return doWithRetry(() -> super.downloadAsTempFile(fileId));
	}

	@Override
	public CompletableFuture<Path> downloadAsTempFile(final String fileId, final Path dir) {
		return doWithRetry(() -> super.downloadAsTempFile(fileId, dir));
	}

	@Override
	public CompletableFuture<Job> getJob(final Job job) {
		return doWithRetry(() -> super.getJob(job));
	}

	@Override
	public CompletableFuture<Void> deleteJob(final Job job) {
		return doWithRetry(() -> super.deleteJob(job));
	}

	@Override
	public CompletableFuture<Job> getJobDone(final Job job) {
		return doWithRetry(() -> super.getJobDone(job));
	}

	@Override
	public CompletableFuture<Job> getJobDone(final String jobId) {
		return doWithRetry(() -> super.getJobDone(jobId));
	}

	private <T> CompletableFuture<T> doWithRetry(final Supplier<CompletableFuture<T>> action) {
		final CompletableFuture<T> cf = new CompletableFuture<>();
		new Retrier<>(cf, action, maxWaitTimeMillis).run();
		return cf;
	}

	private static class Retrier<T> implements Runnable {
		private final Supplier<CompletableFuture<T>> action;
		private final CompletableFuture<T> cf;
		private final AtomicInteger tries;
		private final AtomicReference<Throwable> error;
		private final long maxWaitTimeMillis;

		public Retrier(final CompletableFuture<T> cf, final Supplier<CompletableFuture<T>> action, final long maxWaitTimeMillis) {
			this.cf = cf;
			this.action = action;
			this.tries = new AtomicInteger(0);
			this.error = new AtomicReference<>();
			this.maxWaitTimeMillis = maxWaitTimeMillis;
		}

		@Override
		public void run() {
			tries.incrementAndGet();
			if (tries.get() > 1) {
				LOG.info("Try #{}", tries.get());
			}

			action.get()
			.thenAccept(cf::complete)
			.exceptionally(newError -> {
				final Throwable e = unwrapCompletionException(newError);
				if (!this.error.compareAndSet(null, newRetryFailedException(e))) {
					this.error.get().addSuppressed(e);
				}
				if (LOG.isDebugEnabled()) {
					LOG.debug("Completing retrier with error: {}", this.error.get().toString());
				}
				if (tries.get() < MAX_TRIES && isRetryCandidate(newError)) {
					final long waitTimeMillis = getWaitTimeMillis();
					LOG.info("Triggering try #{} in {} millis after error: {}", tries.get() + 1, waitTimeMillis, newError.toString());
					Delay.delayedExecutor(waitTimeMillis, TimeUnit.MILLISECONDS).execute(this);
				} else {
					cf.completeExceptionally(this.error.get());
				}
				return null;
			});
		}

		private RuntimeException newRetryFailedException(final Throwable newError) {
			if (newError instanceof QetcherRemoteException) {
				return new RetryFailedQetcherRemoteException((QetcherRemoteException) newError);
			} else {
				return new RetryFailedGenericException(newError);
			}
		}

		private Throwable unwrapCompletionException(final Throwable error) {
			if (error instanceof CompletionException && error.getCause() != null) {
				return error.getCause();
			} else {
				return error;
			}
		}

		private long getWaitTimeMillis() {
			return Math.min(
					maxWaitTimeMillis,
					50L * tries.get() + 50L + ThreadLocalRandom.current().nextLong(50));
		}

		private boolean isRetryCandidate(final Throwable error) {
			return
					!Throwables.firstCause(error, RetryFailedException.class).isPresent()
					&& (
							Throwables.firstCause(error, IOException.class)
							.isPresent()
							||
							Throwables.firstCause(error, QetcherRemoteException.class)
							.map(QetcherRemoteException::getStatusCode)
							.map(statusCode -> statusCode == 404 || statusCode == 417) // not found and expectation failed
							.orElse(false)
							);
		}
	}
}
