package de.mklinger.qetcher.client.impl;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Stream;

import de.mklinger.qetcher.client.InputConversionFile;
import de.mklinger.qetcher.client.InputJob;
import de.mklinger.qetcher.client.InputJob.Builder;
import de.mklinger.qetcher.client.model.v1.ConversionFile;
import de.mklinger.qetcher.client.model.v1.MediaType;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class InputJobBuilderImpl implements InputJob.Builder {
	private List<String> conversionFileIds;
	private InputConversionFile inputConversionFile;
	private MediaType fromMediaType;
	private MediaType toMediaType;
	private Duration deleteTimeout;
	private Duration cancelTimeout;
	private String referrer;

	@Override
	public InputJob.Builder fromFile(final String conversionFileId) {
		return fromFiles(conversionFileId);
	}

	@Override
	public InputJob.Builder fromFile(final ConversionFile conversionFile) {
		return fromFile(conversionFile.getFileId());
	}

	@Override
	public Builder fromFiles(final ConversionFile... conversionFiles) {
		return fromFiles(Stream.of(conversionFiles)
				.map(ConversionFile::getFileId)
				.toArray(String[]::new));
	}

	@Override
	public Builder fromFiles(final String... conversionFileIds) {
		this.conversionFileIds = new ArrayList<>(conversionFileIds.length);
		Collections.addAll(this.conversionFileIds, conversionFileIds);
		return this;
	}

	public List<String> getConversionFileIds() {
		return conversionFileIds;
	}

	@Override
	public InputJob.Builder fromFile(final InputConversionFile inputConversionFile) {
		this.inputConversionFile = inputConversionFile;
		return this;
	}

	public InputConversionFile getInputConversionFile() {
		return inputConversionFile;
	}

	@Override
	public InputJob.Builder fromMediaType(final MediaType mediaType) {
		this.fromMediaType = mediaType;
		return this;
	}

	public MediaType getFromMediaType() {
		return fromMediaType;
	}

	@Override
	public InputJob.Builder toMediaType(final MediaType mediaType) {
		this.toMediaType = mediaType;
		return this;
	}

	public MediaType getToMediaType() {
		return toMediaType;
	}

	@Override
	public InputJob.Builder deleteTimeout(final Duration deleteTimeout) {
		this.deleteTimeout = deleteTimeout;
		return this;
	}

	public Duration getDeleteTimeout() {
		return deleteTimeout;
	}

	@Override
	public InputJob.Builder cancelTimeout(final Duration cancelTimeout) {
		this.cancelTimeout = cancelTimeout;
		return this;
	}

	public Duration getCancelTimeout() {
		return cancelTimeout;
	}

	@Override
	public InputJob.Builder referrer(final String referrer) {
		this.referrer = referrer;
		return this;
	}

	public String getReferrer() {
		return referrer;
	}

	@Override
	public InputJob build() {
		return InstanceFactory.newInstance(this);
	}
}