package de.mklinger.qetcher.client.model.v1.jackson;

import java.lang.annotation.Annotation;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.annotation.JsonPOJOBuilder;
import com.fasterxml.jackson.databind.annotation.JsonPOJOBuilder.Value;
import com.fasterxml.jackson.databind.introspect.AnnotatedClass;
import com.fasterxml.jackson.databind.introspect.JacksonAnnotationIntrospector;

import de.mklinger.qetcher.client.model.v1.V1;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class QetcherJacksonAnnotationIntrospector extends JacksonAnnotationIntrospector {
	private static final long serialVersionUID = 1L;

	private static final Logger LOG = LoggerFactory.getLogger(QetcherJacksonAnnotationIntrospector.class);

	private static final String MODEL_PACKAGE_NAME = V1.class.getPackage().getName();
	private static final String MODEL_PACKAGE_SUB_PREFIX = MODEL_PACKAGE_NAME + ".";
	private static final String MODEL_BUILDER_PACKAGE = MODEL_PACKAGE_SUB_PREFIX + "builder.";

	@Override
	public Class<?> findPOJOBuilder(final AnnotatedClass ac) {
		Class<?> builder = super.findPOJOBuilder(ac);
		if (builder == null && isModelPackageClass(ac)) {
			// super did not find a builder (by annotation), try to find a "Builder" in dedicated builder package
			final String innerBuilderClassName = getBuilderClassName(ac);
			try {
				builder = ac.getRawType().getClassLoader().loadClass(innerBuilderClassName);
			} catch (final ClassNotFoundException e) {
				LOG.debug("Builder class not found: {}", innerBuilderClassName, e);
				// ignore
			}
		}
		if (builder == null) {
			LOG.debug("No builder found for {}", ac.getRawType());
		} else {
			LOG.debug("Using builder {} for {}", builder, ac.getRawType());
		}
		return builder;
	}

	private String getBuilderClassName(final AnnotatedClass ac) {
		String simpleClassName = ac.getRawType().getSimpleName();
		if (simpleClassName.endsWith("Impl")) {
			simpleClassName = simpleClassName.substring(0, simpleClassName.length() - "Impl".length());
		}
		return MODEL_BUILDER_PACKAGE + simpleClassName + "Builder";
	}

	private static final JsonPOJOBuilder DEFAULT_POJO_BUILDER = new JsonPOJOBuilder() {
		@Override
		public Class<? extends Annotation> annotationType() {
			return getClass();
		}

		@Override
		public String withPrefix() {
			return "";
		}

		@Override
		public String buildMethodName() {
			return "build";
		}
	};

	@Override
	public Value findPOJOBuilderConfig(final AnnotatedClass ac) {
		Value value = super.findPOJOBuilderConfig(ac);
		if (value == null && isModelPackageClass(ac)) {
			// if annotated class is in model package or a sub-package thereof, use qetcher defaults
			value = new Value(DEFAULT_POJO_BUILDER);
		}
		return value;
	}

	private boolean isModelPackageClass(final AnnotatedClass ac) {
		final Package pakkage = ac.getRawType().getPackage();
		if (pakkage == null) {
			return false;
		}
		final String packageName = pakkage.getName();
		return packageName.equals(MODEL_PACKAGE_NAME)
				|| packageName.startsWith(MODEL_PACKAGE_SUB_PREFIX);
	}
}
