package de.mklinger.qetcher.client.model.v1.jackson.xml;

import java.util.Collection;

import com.fasterxml.jackson.databind.PropertyName;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.cfg.MapperConfig;
import com.fasterxml.jackson.databind.introspect.Annotated;
import com.fasterxml.jackson.databind.introspect.AnnotatedClass;
import com.fasterxml.jackson.databind.introspect.AnnotatedField;
import com.fasterxml.jackson.databind.introspect.AnnotatedMethod;
import com.fasterxml.jackson.dataformat.xml.JacksonXmlAnnotationIntrospector;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de - klingerm
 */
public class QetcherJacksonXmlAnnotationIntrospector extends JacksonXmlAnnotationIntrospector {
	private static final long serialVersionUID = 1L;

	private static final String QETCHER_NAMESPACE = "http://mklinger.de/xml/qetcher";

	public QetcherJacksonXmlAnnotationIntrospector() {
		super(false);
	}

	@Override
	public PropertyName findRootName(final AnnotatedClass ac) {
		PropertyName rootName = super.findRootName(ac);
		if (rootName == null) {
			String localName = ac.getRawType().getSimpleName();
			if (localName.length() > 1) {
				localName = localName.substring(0, 1).toLowerCase() + localName.substring(1);
			}
			if (localName.length() > "Impl".length() && localName.endsWith("Impl")) {
				localName = localName.substring(0, localName.length() - "Impl".length());
			}

			rootName = PropertyName.construct(localName, QETCHER_NAMESPACE);
		}
		return rootName;
	}

	@Override
	public String findNamespace(final Annotated ann) {
		String namespace = super.findNamespace(ann);
		if (namespace == null) {
			namespace = QETCHER_NAMESPACE;
		}
		return namespace;
	}

	@Override
	public Object findNamingStrategy(final AnnotatedClass ac) {
		return new PropertyNamingStrategy() {
			private static final long serialVersionUID = 1L;

			@Override
			public String nameForField(final MapperConfig<?> config, final AnnotatedField field, final String defaultName) {
				if (isPlural(field.getRawType(), defaultName)) {
					return getSingular(defaultName);
				}
				return super.nameForField(config, field, defaultName);
			}

			@Override
			public String nameForGetterMethod(final MapperConfig<?> config, final AnnotatedMethod method, final String defaultName) {
				if (isPlural(method.getRawReturnType(), defaultName)) {
					return getSingular(defaultName);
				}
				return super.nameForGetterMethod(config, method, defaultName);
			}

			@Override
			public String nameForSetterMethod(final MapperConfig<?> config, final AnnotatedMethod method, final String defaultName) {
				if (isPlural(method.getRawParameterType(0), defaultName)) {
					return getSingular(defaultName);
				}
				return super.nameForSetterMethod(config, method, defaultName);
			}

			private boolean isPlural(final Class<?> clazz, final String defaultName) {
				return isCollection(clazz) && defaultName.length() > 1 && defaultName.endsWith("s");
			}

			private String getSingular(final String defaultName) {
				if (defaultName.endsWith("ies")) {
					return defaultName.substring(0, defaultName.length() - 3) + "y";
				} else {
					return defaultName.substring(0, defaultName.length() - 1);
				}
			}

			private boolean isCollection(final Class<?> clazz) {
				return Collection.class.isAssignableFrom(clazz) || clazz.isArray();
			}
		};
	}
}
