package de.mklinger.qetcher.client.model.v1.impl;

import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Pattern;

import de.mklinger.micro.lists.Lists;
import de.mklinger.micro.maps.Maps;
import de.mklinger.qetcher.client.model.v1.Error;
import de.mklinger.qetcher.client.model.v1.Job;
import de.mklinger.qetcher.client.model.v1.JobState;
import de.mklinger.qetcher.client.model.v1.MediaType;
import de.mklinger.qetcher.client.model.v1.ModelValidationException;
import de.mklinger.qetcher.client.model.v1.builder.JobBuilder;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class JobImpl implements Job {
	/**
	 * Property names must be valid XML element names to allow Jackson XML serialization.
	 *
	 * See https://www.w3.org/TR/2008/REC-xml-20081126/#NT-Name
	 * [4]   	NameStartChar	   ::=   	":" | [A-Z] | "_" | [a-z] | [#xC0-#xD6] | [#xD8-#xF6] | [#xF8-#x2FF] | [#x370-#x37D] | [#x37F-#x1FFF] | [#x200C-#x200D] | [#x2070-#x218F] | [#x2C00-#x2FEF] | [#x3001-#xD7FF] | [#xF900-#xFDCF] | [#xFDF0-#xFFFD] | [#x10000-#xEFFFF]
	 * [4a]   	NameChar	   ::=   	NameStartChar | "-" | "." | [0-9] | #xB7 | [#x0300-#x036F] | [#x203F-#x2040]
	 * [5]   	Name	   ::=   	NameStartChar (NameChar)*
	 *
	 * To ease validation, let's be even more restrictive.
	 */
	private static final Pattern PROPERTY_KEY_PATTERN = Pattern.compile("[a-z_A-Z][a-z_A-Z\\-\\.0-9]*");

	private final String jobId;
	private final String parentJobId;
	private final String tenantId;
	private final String nodeId;
	private final String converterId;
	private final int priority;
	private final MediaType fromMediaType;
	private final List<String> inputFileIds;
	private final MediaType toMediaType;
	private final List<String> resultFileIds;
	private final JobState state;
	private final Error error;
	private final Map<String, String> properties;
	private final Instant created;
	private final Instant updated;
	private final Duration estimatedRunTime;
	private final String referrer;
	private final Duration cancelTimeout;
	private final Duration deleteTimeout;

	public JobImpl(final JobBuilder builder) {
		this.jobId = builder.getJobId();
		this.parentJobId = builder.getParentJobId();
		this.tenantId = builder.getTenantId();
		this.nodeId = builder.getNodeId();
		this.converterId = builder.getConverterId();
		this.priority = builder.getPriority();
		this.fromMediaType = builder.getFromMediaType();
		this.inputFileIds = Lists.newImmutableList(builder.getInputFileIds());
		this.toMediaType = builder.getToMediaType();
		this.resultFileIds = Lists.newImmutableList(builder.getResultFileIds());
		this.state = builder.getState();
		this.error = builder.getError();
		this.properties = Maps.newImmutableMap(builder.getProperties());
		this.created = builder.getCreated();
		if (builder.getUpdated() == null) {
			this.updated = Instant.now();
		} else {
			this.updated = builder.getUpdated();
		}
		this.estimatedRunTime = builder.getEstimatedRunTime();
		this.referrer = builder.getReferrer();
		this.cancelTimeout = builder.getCancelTimeout();
		this.deleteTimeout = builder.getDeleteTimeout();

		try {
			validate();
		} catch (final Exception e) {
			throw new ModelValidationException(Job.class, e);
		}
	}

	private void validate() {
		Objects.requireNonNull(jobId);
		Objects.requireNonNull(tenantId);
		Objects.requireNonNull(priority);
		Objects.requireNonNull(fromMediaType);
		Objects.requireNonNull(inputFileIds);
		if (inputFileIds.isEmpty()) {
			throw new IllegalArgumentException();
		}
		Objects.requireNonNull(toMediaType);
		Objects.requireNonNull(resultFileIds);
		Objects.requireNonNull(state);
		Objects.requireNonNull(properties);
		Objects.requireNonNull(created);
		Objects.requireNonNull(updated);
		if (error == null && state == JobState.ERROR) {
			throw new IllegalArgumentException("Job state is error, but error is not set");
		}

		for (final String propertyKey : properties.keySet()) {
			if (!PROPERTY_KEY_PATTERN.matcher(propertyKey).matches()) {
				throw new IllegalArgumentException("Illegal property key. Property keys must match the regular expression " + PROPERTY_KEY_PATTERN.pattern());
			}
		}
	}

	@Override
	public String getJobId() {
		return jobId;
	}

	@Override
	public String getParentJobId() {
		return parentJobId;
	}

	@Override
	public String getTenantId() {
		return tenantId;
	}

	@Override
	public String getNodeId() {
		return nodeId;
	}

	@Override
	public String getConverterId() {
		return converterId;
	}

	@Override
	public int getPriority() {
		return priority;
	}

	@Override
	public MediaType getFromMediaType() {
		return fromMediaType;
	}

	@Override
	public List<String> getInputFileIds() {
		return inputFileIds;
	}

	@Override
	public MediaType getToMediaType() {
		return toMediaType;
	}

	@Override
	public List<String> getResultFileIds() {
		return resultFileIds;
	}

	@Override
	public JobState getState() {
		return state;
	}

	@Override
	public Error getError() {
		return error;
	}

	@Override
	public Map<String, String> getProperties() {
		return properties;
	}

	@Override
	public Instant getCreated() {
		return created;
	}

	@Override
	public Instant getUpdated() {
		return updated;
	}

	@Override
	public Duration getEstimatedRunTime() {
		return estimatedRunTime;
	}

	@Override
	public String getReferrer() {
		return referrer;
	}

	@Override
	public Duration getCancelTimeout() {
		return cancelTimeout;
	}

	@Override
	public Duration getDeleteTimeout() {
		return deleteTimeout;
	}

	@Override
	public String toString() {
		return "JobImpl{" +
				"jobId='" + jobId + '\'' +
				", parentJobId='" + parentJobId + '\'' +
				", tenantId='" + tenantId + '\'' +
				", nodeId='" + nodeId + '\'' +
				", converterId='" + converterId + '\'' +
				", priority=" + priority +
				", fromMediaType=" + fromMediaType +
				", toMediaType=" + toMediaType +
				", state=" + state +
				", error=" + error +
				'}';
	}

}