package de.mklinger.qetcher.client.common;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.TreeMap;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class Maps {
	private static final int MIN_INITIAL_CAPACITY = 4;

	/** No instantiation */
	private Maps() {}

	/**
	 * Returns a HashMap with a capacity that is sufficient to keep the map
	 * from being resized as long as it grows no larger than expectedSize
	 * with the default load factor (0.75).
	 */
	public static <K, V> HashMap<K, V> newHashMap(final int expectedSize) {
		// See code in java.util.HashSet.HashSet(Collection<? extends E>)
		return new HashMap<>(Math.max((int) (expectedSize / .75f) + 1, MIN_INITIAL_CAPACITY));
	}

	public static <K, V> Map<K, V> newImmutableMap(final Map<K, V> original) {
		if (original == null || original.isEmpty()) {
			return Collections.emptyMap();
		}
		if (original.size() == 1) {
			final Entry<K, V> entry = original.entrySet().iterator().next();
			return Collections.singletonMap(entry.getKey(), entry.getValue());
		}
		return Collections.unmodifiableMap(new HashMap<>(original));
	}

	public static HashMap<String, String> newHashMap(final String... s) {
		Objects.requireNonNull(s);
		if (s.length % 2 != 0) {
			throw new IllegalArgumentException();
		}
		final HashMap<String, String> m = newHashMap(s.length / 2);
		for (int i = 0; i < s.length; i+=2) {
			m.put(s[i], s[i + 1]);
		}
		return m;
	}

	@SuppressWarnings("unchecked")
	public static <K, V> HashMap<K, V> newHashMap(final K key1, final V value1, final Object... others) {
		if (others != null && others.length > 0) {
			if (others.length % 2 != 0) {
				throw new IllegalArgumentException();
			}
			final HashMap<K, V> m = newHashMap(others.length / 2 + 1);
			m.put(key1, value1);
			for (int i = 0; i < others.length; i+=2) {
				m.put((K)others[i], (V)others[i + 1]);
			}
			return m;
		} else {
			final HashMap<K, V> m = newHashMap(1);
			m.put(key1, value1);
			return m;
		}
	}

	public static Map<String, String> newHashMapFrom(final Map<String, String> original, final String... s) {
		Objects.requireNonNull(original);
		Objects.requireNonNull(s);
		if (s.length % 2 != 0) {
			throw new IllegalArgumentException();
		}
		final Map<String, String> m = newHashMap(original.size() + s.length / 2);
		m.putAll(original);
		for (int i = 0; i < s.length; i+=2) {
			m.put(s[i], s[i + 1]);
		}
		return m;
	}

	public static Map<String, String> newTreeMap(final String... s) {
		Objects.requireNonNull(s);
		if (s.length % 2 != 0) {
			throw new IllegalArgumentException();
		}
		final Map<String, String> m = new TreeMap<>();
		for (int i = 0; i < s.length; i+=2) {
			m.put(s[i], s[i + 1]);
		}
		return m;
	}
}
