/*
 * Licensed to Elasticsearch under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package de.mklinger.qetcher.client.common;

import java.util.HashMap;
import java.util.Map;

/**
 *
 */
public class LoggerMessageFormat {
	private static final char DELIM_START = '{';
	private static final String DELIM_STR = "{}";
	private static final char ESCAPE_CHAR = '\\';

	/** No instantiation */
	private LoggerMessageFormat() {}

	public static String format(final String messagePattern, final Object... argArray) {
		if (messagePattern == null) {
			return null;
		}
		if (argArray == null) {
			return messagePattern;
		}
		int i = 0;
		int j;
		final StringBuilder sbuf = new StringBuilder(messagePattern.length() + 50);

		for (int L = 0; L < argArray.length; L++) {

			j = messagePattern.indexOf(DELIM_STR, i);

			if (j == -1) {
				// no more variables
				if (i == 0) { // this is a simple string
					return messagePattern;
				} else { // add the tail string which contains no variables and return
					// the result.
					sbuf.append(messagePattern.substring(i, messagePattern.length()));
					return sbuf.toString();
				}
			} else {
				if (isEscapedDelimiter(messagePattern, j)) {
					if (!isDoubleEscaped(messagePattern, j)) {
						L--; // DELIM_START was escaped, thus should not be incremented
						sbuf.append(messagePattern.substring(i, j - 1));
						sbuf.append(DELIM_START);
						i = j + 1;
					} else {
						// The escape character preceding the delimiter start is
						// itself escaped: "abc x:\\{}"
						// we have to consume one backward slash
						sbuf.append(messagePattern.substring(i, j - 1));
						deeplyAppendParameter(sbuf, argArray[L], new HashMap<>());
						i = j + 2;
					}
				} else {
					// normal case
					sbuf.append(messagePattern.substring(i, j));
					deeplyAppendParameter(sbuf, argArray[L], new HashMap<>());
					i = j + 2;
				}
			}
		}
		// append the characters following the last {} pair.
		sbuf.append(messagePattern.substring(i, messagePattern.length()));
		return sbuf.toString();
	}

	static boolean isEscapedDelimiter(final String messagePattern,
			final int delimiterStartIndex) {

		if (delimiterStartIndex == 0) {
			return false;
		}
		final char potentialEscape = messagePattern.charAt(delimiterStartIndex - 1);
		if (potentialEscape == ESCAPE_CHAR) {
			return true;
		} else {
			return false;
		}
	}

	static boolean isDoubleEscaped(final String messagePattern, final int delimiterStartIndex) {
		if (delimiterStartIndex >= 2 && messagePattern.charAt(delimiterStartIndex - 2) == ESCAPE_CHAR) {
			return true;
		} else {
			return false;
		}
	}

	private static void deeplyAppendParameter(final StringBuilder sbuf, final Object o, final Map<Object, Object> seenMap) {
		if (o == null) {
			sbuf.append("null");
			return;
		}
		if (!o.getClass().isArray()) {
			safeObjectAppend(sbuf, o);
		} else {
			// check for primitive array types because they
			// unfortunately cannot be cast to Object[]
			if (o instanceof boolean[]) {
				booleanArrayAppend(sbuf, (boolean[]) o);
			} else if (o instanceof byte[]) {
				byteArrayAppend(sbuf, (byte[]) o);
			} else if (o instanceof char[]) {
				charArrayAppend(sbuf, (char[]) o);
			} else if (o instanceof short[]) {
				shortArrayAppend(sbuf, (short[]) o);
			} else if (o instanceof int[]) {
				intArrayAppend(sbuf, (int[]) o);
			} else if (o instanceof long[]) {
				longArrayAppend(sbuf, (long[]) o);
			} else if (o instanceof float[]) {
				floatArrayAppend(sbuf, (float[]) o);
			} else if (o instanceof double[]) {
				doubleArrayAppend(sbuf, (double[]) o);
			} else {
				objectArrayAppend(sbuf, (Object[]) o, seenMap);
			}
		}
	}

	private static void safeObjectAppend(final StringBuilder sbuf, final Object o) {
		try {
			final String oAsString = o.toString();
			sbuf.append(oAsString);
		} catch (final Throwable t) {
			sbuf.append("[FAILED toString()]");
		}

	}

	private static void objectArrayAppend(final StringBuilder sbuf, final Object[] a, final Map<Object, Object> seenMap) {
		sbuf.append('[');
		if (!seenMap.containsKey(a)) {
			seenMap.put(a, null);
			final int len = a.length;
			for (int i = 0; i < len; i++) {
				deeplyAppendParameter(sbuf, a[i], seenMap);
				if (i != len - 1) {
					sbuf.append(", ");
				}
			}
			// allow repeats in siblings
			seenMap.remove(a);
		} else {
			sbuf.append("...");
		}
		sbuf.append(']');
	}

	private static void booleanArrayAppend(final StringBuilder sbuf, final boolean[] a) {
		sbuf.append('[');
		final int len = a.length;
		for (int i = 0; i < len; i++) {
			sbuf.append(a[i]);
			if (i != len - 1) {
				sbuf.append(", ");
			}
		}
		sbuf.append(']');
	}

	private static void byteArrayAppend(final StringBuilder sbuf, final byte[] a) {
		sbuf.append('[');
		final int len = a.length;
		for (int i = 0; i < len; i++) {
			sbuf.append(a[i]);
			if (i != len - 1) {
				sbuf.append(", ");
			}
		}
		sbuf.append(']');
	}

	private static void charArrayAppend(final StringBuilder sbuf, final char[] a) {
		sbuf.append('[');
		final int len = a.length;
		for (int i = 0; i < len; i++) {
			sbuf.append(a[i]);
			if (i != len - 1) {
				sbuf.append(", ");
			}
		}
		sbuf.append(']');
	}

	private static void shortArrayAppend(final StringBuilder sbuf, final short[] a) {
		sbuf.append('[');
		final int len = a.length;
		for (int i = 0; i < len; i++) {
			sbuf.append(a[i]);
			if (i != len - 1) {
				sbuf.append(", ");
			}
		}
		sbuf.append(']');
	}

	private static void intArrayAppend(final StringBuilder sbuf, final int[] a) {
		sbuf.append('[');
		final int len = a.length;
		for (int i = 0; i < len; i++) {
			sbuf.append(a[i]);
			if (i != len - 1) {
				sbuf.append(", ");
			}
		}
		sbuf.append(']');
	}

	private static void longArrayAppend(final StringBuilder sbuf, final long[] a) {
		sbuf.append('[');
		final int len = a.length;
		for (int i = 0; i < len; i++) {
			sbuf.append(a[i]);
			if (i != len - 1) {
				sbuf.append(", ");
			}
		}
		sbuf.append(']');
	}

	private static void floatArrayAppend(final StringBuilder sbuf, final float[] a) {
		sbuf.append('[');
		final int len = a.length;
		for (int i = 0; i < len; i++) {
			sbuf.append(a[i]);
			if (i != len - 1) {
				sbuf.append(", ");
			}
		}
		sbuf.append(']');
	}

	private static void doubleArrayAppend(final StringBuilder sbuf, final double[] a) {
		sbuf.append('[');
		final int len = a.length;
		for (int i = 0; i < len; i++) {
			sbuf.append(a[i]);
			if (i != len - 1) {
				sbuf.append(", ");
			}
		}
		sbuf.append(']');
	}
}
