package de.mklinger.qetcher.client.model.v1.builder;

import java.math.BigInteger;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;

import de.mklinger.qetcher.client.model.v1.ConversionFile;
import de.mklinger.qetcher.client.model.v1.MediaType;
import de.mklinger.qetcher.client.model.v1.impl.ConversionFileImpl;

/**
 * Created by Marius Heinzmann on 3/26/18.
 *
 * @author Marius Heinzmann - mheinzmann[at]mklinger[dot]de
 */
public class ConversionFileBuilder {
	private String fileId;
	private boolean serviceManaged;
	private String tenantId;
	private String name;
	private BigInteger size;
	private MediaType mediaType;
	private List<String> nodeIds;
	private Duration deleteTimeout;
	private Instant created;
	private Instant updated;

	public static ConversionFileBuilder of(final ConversionFile conversionFile) {
		return new ConversionFileBuilder()
				.fileId(conversionFile.getFileId())
				.serviceManaged(conversionFile.isServiceManaged())
				.tenantId(conversionFile.getTenantId())
				.name(conversionFile.getName())
				.size(conversionFile.getSize())
				.mediaType(conversionFile.getMediaType())
				.nodeIds(conversionFile.getNodeIds())
				.deleteTimeout(conversionFile.getDeleteTimeout())
				.created(conversionFile.getCreated())
				.updated(Instant.now());
	}

	public ConversionFileBuilder fileId(final String fileId) {
		this.fileId = fileId;
		return this;
	}

	public ConversionFileBuilder serviceManaged(final boolean serviceManaged) {
		this.serviceManaged = serviceManaged;
		return this;
	}

	public ConversionFileBuilder tenantId(final String tenantId) {
		this.tenantId = tenantId;
		return this;
	}

	public ConversionFileBuilder name(final String name) {
		this.name = name;
		return this;
	}

	public ConversionFileBuilder size(final BigInteger size) {
		this.size = size;
		return this;
	}

	public ConversionFileBuilder mediaType(final MediaType mediaType) {
		this.mediaType = mediaType;
		return this;
	}

	public ConversionFileBuilder nodeIds(final List<String> nodeIds) {
		if (nodeIds == null || nodeIds.isEmpty()) {
			this.nodeIds = null;
		} else {
			this.nodeIds = new ArrayList<>(nodeIds);
		}
		return this;
	}

	public ConversionFileBuilder withNodeId(final String nodeId) {
		if (this.nodeIds == null) {
			this.nodeIds = new ArrayList<>();
		}
		if (!this.nodeIds.contains(nodeId)) {
			this.nodeIds.add(nodeId);
		}
		return this;
	}

	public ConversionFileBuilder withoutNodeId(final String nodeId) {
		if (this.nodeIds == null) {
			return this;
		}
		this.nodeIds.remove(nodeId);
		return this;
	}

	public ConversionFileBuilder deleteTimeout(final Duration deleteTimeout) {
		this.deleteTimeout = deleteTimeout;
		return this;
	}

	public ConversionFileBuilder created(final Instant created) {
		this.created = created;
		return this;
	}

	public ConversionFileBuilder updated(final Instant updated) {
		this.updated = updated;
		return this;
	}

	public ConversionFile build() {
		return new ConversionFileImpl(this);
	}

	public String getFileId() {
		return fileId;
	}

	public boolean isServiceManaged() {
		return serviceManaged;
	}

	public String getTenantId() {
		return tenantId;
	}

	public String getName() {
		return name;
	}

	public BigInteger getSize() {
		return size;
	}

	public MediaType getMediaType() {
		return mediaType;
	}

	public List<String> getNodeIds() {
		return nodeIds;
	}

	public Duration getDeleteTimeout() {
		return deleteTimeout;
	}

	public Instant getCreated() {
		return created;
	}

	public Instant getUpdated() {
		return updated;
	}
}
