package de.mklinger.qetcher.client.impl.retry;

import java.io.File;
import java.io.IOException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import de.mklinger.micro.annotations.VisibleForTesting;
import de.mklinger.micro.throwables.Throwables;
import de.mklinger.qetcher.client.InputConversionFile;
import de.mklinger.qetcher.client.InputJob;
import de.mklinger.qetcher.client.QetcherRemoteException;
import de.mklinger.qetcher.client.impl.QetcherClientBuilderImpl;
import de.mklinger.qetcher.client.impl.QetcherClientImpl;
import de.mklinger.qetcher.client.impl.lookup.ServiceUriSupplier;
import de.mklinger.qetcher.client.model.v1.AvailableConversion;
import de.mklinger.qetcher.client.model.v1.AvailableNode;
import de.mklinger.qetcher.client.model.v1.ConversionFile;
import de.mklinger.qetcher.client.model.v1.Job;

/**
 * Retry on 404 and 417 responses and IOExceptions.
 *
 * This can not be implemented as a pure decorator with delegate on top of
 * QetcherClient interface, as we also have to wrap calls made internally,
 * especially calls initiated by {@link #getJobDone(String)}.
 *
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class RetryingQetcherClientImpl extends QetcherClientImpl {
	private static final int MAX_TRIES = 10;
	private static final int DEFAULT_MAX_WAIT_TIME_MILLIS = 1000;
	private final long maxWaitTimeMillis;

	public RetryingQetcherClientImpl(final QetcherClientBuilderImpl builder, final ServiceUriSupplier serviceUriSupplier) {
		this(builder, serviceUriSupplier, DEFAULT_MAX_WAIT_TIME_MILLIS);
	}

	@VisibleForTesting
	protected RetryingQetcherClientImpl(final QetcherClientBuilderImpl builder, final ServiceUriSupplier serviceUriSupplier, final long maxWaitTimeMillis) {
		super(builder, serviceUriSupplier);
		this.maxWaitTimeMillis = maxWaitTimeMillis;
	}

	@Override
	public CompletableFuture<ConversionFile> uploadFile(final InputConversionFile inputFile) {
		return doWithRetry(() -> super.uploadFile(inputFile));
	}

	@Override
	public CompletableFuture<ConversionFile> getFile(final String fileId) {
		return doWithRetry(() -> super.getFile(fileId));
	}

	@Override
	public CompletableFuture<List<ConversionFile>> getFiles() {
		return doWithRetry(() -> super.getFiles());
	}

	@Override
	public CompletableFuture<Void> deleteFile(final String fileId) {
		return doWithRetry(() -> super.deleteFile(fileId));
	}

	@Override
	public CompletableFuture<Path> downloadAsFile(final String fileId, final Path file, final OpenOption... openOptions) {
		return doWithRetry(() -> super.downloadAsFile(fileId, file, openOptions));
	}

	@Override
	public CompletableFuture<Job> createJob(final InputJob inputJob) {
		return doWithRetry(() -> super.createJob(inputJob));
	}

	@Override
	public CompletableFuture<Job> getJob(final String jobId) {
		return doWithRetry(() -> super.getJob(jobId));
	}

	@Override
	public CompletableFuture<List<Job>> getJobs() {
		return doWithRetry(() -> super.getJobs());
	}

	@Override
	public CompletableFuture<Void> deleteJob(final String jobId) {
		return doWithRetry(() -> super.deleteJob(jobId));
	}

	@Override
	public CompletableFuture<List<AvailableConversion>> getAvailableConversions() {
		return doWithRetry(() -> super.getAvailableConversions());
	}

	@Override
	public CompletableFuture<List<AvailableNode>> getAvailableNodes() {
		return doWithRetry(() -> super.getAvailableNodes());
	}

	@Override
	public CompletableFuture<ConversionFile> getFile(final ConversionFile file) {
		return doWithRetry(() -> super.getFile(file));
	}

	@Override
	public CompletableFuture<Void> deleteFile(final ConversionFile file) {
		return doWithRetry(() -> super.deleteFile(file));
	}

	@Override
	public CompletableFuture<Path> downloadAsFile(final String fileId, final Path file) {
		return doWithRetry(() -> super.downloadAsFile(fileId, file));
	}

	@Override
	public CompletableFuture<File> downloadAsFile(final String fileId, final File file) {
		return doWithRetry(() -> super.downloadAsFile(fileId, file));
	}

	@Override
	public CompletableFuture<Path> downloadAsTempFile(final String fileId) {
		return doWithRetry(() -> super.downloadAsTempFile(fileId));
	}

	@Override
	public CompletableFuture<Path> downloadAsTempFile(final String fileId, final Path dir) {
		return doWithRetry(() -> super.downloadAsTempFile(fileId, dir));
	}

	@Override
	public CompletableFuture<Job> getJob(final Job job) {
		return doWithRetry(() -> super.getJob(job));
	}

	@Override
	public CompletableFuture<Void> deleteJob(final Job job) {
		return doWithRetry(() -> super.deleteJob(job));
	}

	@Override
	public CompletableFuture<Job> getJobDone(final Job job) {
		return doWithRetry(() -> super.getJobDone(job));
	}

	@Override
	public CompletableFuture<Job> getJobDone(final String jobId) {
		return doWithRetry(() -> super.getJobDone(jobId));
	}

	private <T> CompletableFuture<T> doWithRetry(final Supplier<CompletableFuture<T>> action) {
		final CompletableFuture<T> cf = new CompletableFuture<>();
		new Retrier<>(cf, action, this::isRetryCandidate, MAX_TRIES, maxWaitTimeMillis).run();
		return cf;
	}

	private boolean isRetryCandidate(final Throwable error) {
		return
				!isRetryFailedException(error)
				&& (
						Throwables.firstCause(error, IOException.class)
						.isPresent()
						||
						Throwables.firstCause(error, QetcherRemoteException.class)
						.map(QetcherRemoteException::getStatusCode)
						.map(statusCode -> statusCode == 404 || statusCode == 417) // not found and expectation failed
						.orElse(false)
						);
	}

	private boolean isRetryFailedException(final Throwable error) {
		return Throwables.firstCause(error, RetryFailedException.class).isPresent();
	}
}
