package de.mklinger.qetch.client.impl;

import java.net.URI;
import java.security.KeyStore;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import de.mklinger.qetch.client.QetchClient;

/**
 * @author Marc Klinger - mklinger[at]mklinger[dot]de
 */
public class QetchClientBuilderImpl implements QetchClient.Builder {
	private List<URI> serviceUris;
	private boolean serviceLookupDisabled;
	private KeyStore trustStore;
	private KeyStore keyStore;
	private String keyPassword;

	@Override
	public QetchClient.Builder serviceAddresses(final String... serviceAddresses) {
		this.serviceUris = new ArrayList<>(Arrays.stream(serviceAddresses)
				.map(s -> "https://" + s)
				.map(URI::create)
				.collect(Collectors.toList()));
		return this;
	}

	@Override
	public QetchClient.Builder withServiceAddress(final String serviceAddress) {
		if (serviceUris == null) {
			serviceUris = new ArrayList<>();
		}
		serviceUris.add(URI.create("https://" + serviceAddress));
		return this;
	}

	@Override
	public QetchClient.Builder withServiceAddress(final String host, final int port) {
		return withServiceAddress(Objects.requireNonNull(host) + ":" + requireValidPort(port));
	}

	private static int requireValidPort(final int port) {
		if (port <= 0) {
			throw new IllegalArgumentException("Invalid port: " + port);
		}
		return port;
	}

	public List<URI> getServiceUris() {
		return serviceUris;
	}

	@Override
	public QetchClient.Builder disableServiceLookup() {
		this.serviceLookupDisabled = true;
		return this;
	}

	public boolean isServiceLookupDisabled() {
		return this.serviceLookupDisabled;
	}

	@Override
	public QetchClient.Builder trustStore(final KeyStore trustStore) {
		this.trustStore = trustStore;
		return this;
	}

	public KeyStore getTrustStore() {
		return trustStore;
	}

	@Override
	public QetchClient.Builder keyStore(final KeyStore keyStore, final String keyPassword) {
		this.keyStore = keyStore;
		this.keyPassword = keyPassword;
		return this;
	}

	public KeyStore getKeyStore() {
		return keyStore;
	}

	public String getKeyPassword() {
		return keyPassword;
	}

	@Override
	public QetchClient build() {
		return InstanceFactory.newInstance(this);
	}
}
